import { forwardRef } from 'react';
import clsx from 'clsx';

import { HTMLElementType, RefOf } from '../../types';

import type {
    TextComponent,
    TextLeading,
    TextLeadingBySize,
    TextProps,
    TextWeight,
    TextWeightBySize,
} from './text.types';
import { TextSize } from './text.types';

/**
 * @see https://www.figma.com/file/I8EBpOVkBHKTHVrDyWuaaIhI/Travel.Styles?node-id=4169%3A1
 */

export const Text: TextComponent = forwardRef(
    (
        {
            children,
            className,
            testId,
            size = 'sm',
            weight = TEXT_DEFAULT_WEIGHT[size],
            leading = TEXT_DEFAULT_LEADING[size],
            as: asProp,
            ...props
        }: TextProps<'span', 'sm'>,
        ref: RefOf<'span'>,
    ) => {
        const Component = (asProp || tags[size] || 'span') as 'span';

        return (
            <Component
                ref={ref as any}
                data-qa={testId}
                className={clsx(
                    sizeClass[size],
                    weightClass[weight],
                    leadingClass[leading],
                    className,
                )}
                {...props}
            >
                {children}
            </Component>
        );
    },
) as any;

const tags: Partial<Record<TextSize, HTMLElementType>> = {
    xxl: 'h1',
    xl: 'h2',
    lg: 'h3',
};

const sizeClass: Record<TextSize, string> = {
    xxl: 'text-xxl',
    xl: 'text-xl',
    lg: 'text-lg',
    md: 'text-md',
    sm: 'text-sm',
    xs: 'text-xs',
};
const weightClass: Record<TextWeight, string> = {
    regular: 'font-normal',
    medium: 'font-medium',
    bold: 'font-bold',
};
const leadingClass: Record<TextLeading, string> = {
    xxl: 'leading-xxl',
    xl: 'leading-xl',
    lg: 'leading-lg',
    md: 'leading-md',
    sm: 'leading-sm',
    xs: 'leading-xs',
    xxs: 'leading-xxs',
};

export const TEXT_DEFAULT_WEIGHT: TextWeightBySize = {
    xxl: 'bold',
    xl: 'bold',
    lg: 'medium',
    md: 'regular',
    sm: 'regular',
    xs: 'regular',
};
export const TEXT_DEFAULT_LEADING: TextLeadingBySize = {
    xxl: 'xxl',
    xl: 'xl',
    lg: 'lg',
    md: 'md',
    sm: 'sm',
    xs: 'xxs',
};
