import { Meta, Story } from '@storybook/react/types-6-0';

import { Button } from './button';
import type { ButtonSize, ButtonTextSizeMap, ButtonTheme, SharedButtonProps } from './button.types';
import { Icon, Text } from '../../atoms';

const csf: Meta = {
    title: 'molecules/Button',
    component: Button,
};

const BUTTON_THEMES: ButtonTheme[] = ['primary', 'secondary', 'raised', 'outlined', 'attention'];
const BUTTON_SIZES: ButtonSize[] = ['xl', 'lg', 'md', 'sm'];
const BUTTON_TEXT_SIZES: { [S in ButtonSize]: ButtonTextSizeMap[S][] } = {
    sm: ['sm'],
    md: ['sm', 'md'],
    lg: ['sm', 'md', 'lg'],
    xl: ['md', 'lg', 'xl'],
};

export const sizesAndThemes: Story<SharedButtonProps> = args => (
    <table className="text-left">
        <thead>
            <tr>
                <th className="p-4">Размер</th>
                {BUTTON_THEMES.map(theme => (
                    <th className="p-4" key={theme}>
                        {theme}
                    </th>
                ))}
            </tr>
        </thead>
        <tbody>
            {BUTTON_SIZES.map(size => (
                <tr key={size} className="border-b border-grey-light">
                    <td className="p-4">{size}</td>
                    {BUTTON_THEMES.map(theme => (
                        <td className="p-4" key={theme}>
                            <Button {...args} theme={theme} size={size}>
                                Кнопуля
                            </Button>
                        </td>
                    ))}
                </tr>
            ))}
        </tbody>
    </table>
);

sizesAndThemes.storyName = 'Размеры и темы';
sizesAndThemes.args = {};

export const sizeVariants: Story<SharedButtonProps> = args => (
    <div className="space-y-8">
        {BUTTON_SIZES.map(size => (
            <div className="flex gap-4 items-center" key={size}>
                <Text size="xl" className="flex-0 basis-[80px]">
                    {size}
                </Text>
                <Button
                    {...args}
                    theme="secondary"
                    size={size}
                    startNode={<Icon group="16" name="Location" />}
                />
                <Button
                    {...args}
                    shape="circle"
                    theme="secondary"
                    size={size}
                    startNode={<Icon group="16" name="Location" />}
                />
                <Button
                    {...args}
                    theme="secondary"
                    size={size}
                    startNode={<Icon group="16" name="Location" />}
                    endNode={<Icon group="16" name="Aeroexpress" />}
                >
                    По-умолчанию
                </Button>
                {BUTTON_TEXT_SIZES[size].map(textSize => (
                    <>
                        <Button
                            {...args}
                            theme="secondary"
                            key={textSize}
                            textSize={textSize}
                            size={size}
                        >
                            Кнопуля - {textSize}
                        </Button>
                    </>
                ))}
            </div>
        ))}
    </div>
);

sizeVariants.storyName = 'Вариации размеров';
sizeVariants.args = {};

export const states: Story<SharedButtonProps> = args => (
    <table className="text-left">
        <thead>
            <tr>
                <th className="p-4">Default</th>
                <th className="p-4">Disabled</th>
            </tr>
        </thead>
        <tbody>
            {BUTTON_THEMES.map(theme => (
                <tr className="border-b border-grey-light" key={theme}>
                    <td className="p-4" key={theme}>
                        <Button {...args} theme={theme}>
                            Кнопуля
                        </Button>
                    </td>
                    <td className="p-4" key={theme}>
                        <Button {...args} theme={theme} disabled>
                            Кнопуля
                        </Button>
                    </td>
                </tr>
            ))}
        </tbody>
    </table>
);

states.storyName = 'Состояния';
states.args = {};

export const cases: Story<SharedButtonProps> = () => (
    <div className="space-y-4">
        <div className="flex space-x-4">
            <Button size="sm" startNode={<Icon name="CovidShild" />}>
                Кнопка
            </Button>
            <Button size="sm" endNode={<Icon name="CovidShild" />}>
                Кнопка
            </Button>
            <Button size="sm" startNode={<Icon name="CovidShild" />} />
            <Button size="sm" shape="circle" startNode={<Icon name="CovidShild" />} />
        </div>
        <div className="flex space-x-4">
            <Button startNode={<Icon name="CovidShild" />}>Кнопка</Button>
            <Button endNode={<Icon name="CovidShild" />}>Кнопка</Button>
            <Button startNode={<Icon name="CovidShild" />} />
            <Button shape="circle" startNode={<Icon name="CovidShild" />} />
        </div>
        <div className="flex space-x-4">
            <Button size="lg" startNode={<Icon name="CovidShild" />}>
                Кнопка
            </Button>
            <Button size="lg" endNode={<Icon name="CovidShild" />}>
                Кнопка
            </Button>
            <Button size="lg" startNode={<Icon name="CovidShild" />} />
            <Button size="lg" shape="circle" startNode={<Icon name="CovidShild" />} />
        </div>
        <div className="flex space-x-4">
            <Button size="xl" startNode={<Icon group="16" name="Location" />}>
                Кнопка
            </Button>
            <Button size="xl" endNode={<Icon name="CovidShild" />}>
                Кнопка
            </Button>
            <Button size="xl" startNode={<Icon name="CovidShild" />} />
            <Button size="xl" shape="circle" startNode={<Icon name="CovidShild" />} />
        </div>
    </div>
);

cases.storyName = 'Примеры использования';
cases.args = {};

const Template: Story<SharedButtonProps> = args => <Button {...args} />;

export const sandbox = Template.bind({});

sandbox.storyName = 'Sandbox';
sandbox.args = {};

export default csf;
