import { cloneElement, forwardRef, ReactElement } from 'react';
import clsx from 'clsx';

import { RefOf } from '../../types';

import type {
    ButtonComponent,
    ButtonProps,
    ButtonSize,
    ButtonSizeSettings,
    ButtonTheme,
} from './button.types';
import { Text } from '../../atoms';

export const Button: ButtonComponent = forwardRef(
    (
        {
            as: Component = 'button',
            className,
            shape,
            size = 'md',
            textSize = defaultTextSize[size],
            disabled,
            theme = 'primary',
            children,
            startNode,
            endNode,
            leading,
            weight = 'regular',
            ...props
        }: ButtonProps<'button', 'md'>,
        ref: RefOf<'button'>,
    ) => {
        const noChildren = !children;

        return (
            <Component
                ref={ref as any}
                type="button"
                aria-disabled={disabled}
                disabled={disabled}
                className={clsx(
                    'button-base',
                    shape === 'circle' ? 'rounded-full' : 'rounded-sm',
                    !disabled && interactiveClassesMap[theme],
                    sizesMap[size].base,
                    sizesMap[size].content[noChildren ? 'icon' : 'default'],
                    className,
                )}
                {...props}
            >
                {startNode && getIcon(startNode, size, 'left', noChildren)}
                {children && (
                    <Text
                        as="span"
                        size={textSize}
                        weight={weight}
                        leading={leading}
                        className="inline-flex flex-col justify-center truncate-container"
                    >
                        {children}
                    </Text>
                )}
                {endNode && getIcon(endNode, size, 'right', noChildren)}
            </Component>
        );
    },
) as any;

const interactiveClassesMap: Record<ButtonTheme, string> = {
    primary: clsx(
        'bg-button-primary outline-button-primary-ring',
        'hover:bg-button-primary-hover active:bg-button-primary-active',
    ),
    secondary: clsx(
        'bg-button-secondary',
        'hover:bg-button-secondary-hover active:bg-button-secondary-active',
    ),
    outlined: clsx(
        'border bg-white border-button-outlined-ring',
        'hover:border-button-outlined-ring-hover active:border-button-outlined-ring-hover',
    ),
    raised: clsx(
        'bg-white shadow-object outline',
        'hover:shadow-button-hover focus:shadow-button-hover active:shadow-button-active',
    ),
    attention: clsx(
        'bg-button-alert text-red outline-button-alert-ring',
        'hover:bg-button-alert-hover active:bg-button-alert-active',
    ),
    plain: '',
    plus: '',
};

const defaultIconOffset = {
    left: 'mr-2',
    right: 'ml-2',
};
const sizesMap: Record<ButtonSize, ButtonSizeSettings> = {
    sm: {
        base: 'h-7 text-sm px-2',
        content: {
            icon: 'w-7',
        },
        icon: {
            base: 'h-3 w-3',
            left: 'mr-1',
            right: 'ml-1',
        },
    },
    md: {
        base: 'h-9 text-md',
        content: {
            default: 'px-3',
            icon: 'w-9',
        },
        icon: {
            base: 'h-4 w-4',
            ...defaultIconOffset,
        },
    },
    lg: {
        base: 'h-11 text-md',
        content: {
            default: 'px-4',
            icon: 'w-11',
        },
        icon: {
            base: 'h-4 w-4',
            ...defaultIconOffset,
        },
    },
    xl: {
        base: 'h-13 text-lg',
        content: {
            default: 'px-5',
            icon: 'w-13',
        },
        icon: {
            base: 'h-6 w-6',
            ...defaultIconOffset,
        },
    },
};

const defaultTextSize = {
    sm: 'sm',
    md: 'md',
    lg: 'md',
    xl: 'lg',
} as const;

const getIcon = (node: ReactElement, size: ButtonSize, type: 'left' | 'right', onlyIcon = false) =>
    cloneElement(node, {
        className: clsx(
            node.props.className,
            sizesMap[size].icon.base,
            !onlyIcon && sizesMap[size].icon[type],
        ),
    });
