import { FC, useCallback, useEffect, useRef } from 'react';
import clsx from 'clsx';

import { adjustPosition, getPopoverRectOffset, getTransformOriginStyleByRect } from './lib';
import { debounce } from '../../lib/debounce';
import { getElementOffset } from '../../lib/dom/get-element-offset';

import type {
    IPopoverProps,
    PopoverElementPosition,
    PopoverOrigin,
    PopoverRect,
} from './popover.types';
import Modal, { IModalOverlayProps } from '../../atoms/modal';
import Portal from '../../atoms/portal';
import Transition from '../../atoms/transition';
import Card from '../../atoms/card';

// atomic:component:popover
// atomic:component:popover:type.function
export function Popover({
    testId,
    children,
    marginThreshold = defaultMarginThreshold,
    transformOrigin = defaultTransformOrigin,
    anchorOrigin = defaultAnchorOrigin,
    onClose,
    id,
    open = false,
    className,
    anchorNode,
    ...props
}: IPopoverProps) {
    const paperRef = useRef<HTMLDivElement>(null);

    const getAnchorOffset = useCallback(() => {
        if (!anchorNode) return null;

        const rect = anchorNode.getBoundingClientRect();
        const offset = getPopoverRectOffset(rect, anchorOrigin);
        const anchorRealOffset = getElementOffset(anchorNode);

        return {
            top: anchorRealOffset.top + offset.top,
            left: anchorRealOffset.left + offset.left,
        };
    }, [anchorNode, anchorOrigin.vertical, anchorOrigin.horizontal]);

    const getTransformOffset = useCallback(
        (rect: PopoverRect) => getPopoverRectOffset(rect, transformOrigin),
        [transformOrigin.vertical, transformOrigin.horizontal],
    );

    const getPositionStyle = useCallback(
        (element: HTMLElement) => {
            if (!anchorNode) return null;

            const rect: PopoverRect = {
                width: element.offsetWidth,
                height: element.offsetHeight,
            };
            const anchorOffset = getAnchorOffset()!;
            const transformOffset = getTransformOffset(rect);

            const position = {
                top: anchorOffset.top - transformOffset.top,
                left: anchorOffset.left - transformOffset.left,
            } as PopoverElementPosition;

            position.right = position.left + rect.width;
            position.bottom = position.top + rect.height;

            // Window thresholds taking required margin into account
            const { innerWidth, innerHeight } = window;

            adjustPosition(
                transformOffset,
                position,
                {
                    width: innerWidth,
                    height: innerHeight,
                },
                marginThreshold,
            );

            return {
                top: `${Math.round(position.top)}px`,
                left: `${Math.round(position.left)}px`,
                transformOrigin: getTransformOriginStyleByRect(transformOffset),
            };
        },
        [anchorNode, getAnchorOffset, getTransformOffset, marginThreshold],
    );

    const syncElementStyles = useCallback(() => {
        const element = paperRef.current as any as HTMLElement;
        const position = element && getPositionStyle(element);

        console.log('[sync]', position, element);
        if (!element || !position) return;
        Object.assign(element.style, position);
    }, [getPositionStyle]);

    useEffect(() => {
        if (open) {
            syncElementStyles();
        }
    });
    useEffect(() => {
        if (!open || !anchorNode) return;

        const handleResize = debounce(syncElementStyles);

        window.addEventListener('resize', handleResize);

        return () => {
            handleResize.clear();
            window.removeEventListener('resize', handleResize);
        };
    }, [anchorNode, open, syncElementStyles]);

    return (
        <Portal>
            <Modal open={open} transition OverlayComponent={Overlay} onClose={onClose}>
                <Transition type="slide-down" open={open} onEntering={syncElementStyles}>
                    <Card
                        id={id}
                        ref={paperRef}
                        radius="s"
                        shadow="object"
                        spacing={false}
                        className={clsx(
                            'overflow-x-hidden overflow-y-auto absolute outline-none',
                            'min-w-[16px] min-h-[16px] max-w-[calc(100%-32px)] max-h-[calc(100%-32px)]',
                            className,
                        )}
                        {...props}
                    >
                        {children}
                    </Card>
                </Transition>
            </Modal>
        </Portal>
    );
}

const defaultMarginThreshold = 16;
const defaultAnchorOrigin: PopoverOrigin = {
    vertical: 'end',
    horizontal: 'center',
};
const defaultTransformOrigin: PopoverOrigin = {
    vertical: 'start',
    horizontal: 'center',
};

const Overlay: FC<IModalOverlayProps> = ({ open, onClick }) =>
    open ? <div className="overlay bg-transparent" onClick={onClick} /> : null;
