import clsx from 'clsx';
import type { IRangeProps } from './range.types';
import { useMemo, useRef } from 'react';
import { RangeControl } from '../../atoms';
import { normalizeRangeValue } from './lib';
import { useDraggable } from '../../hooks';

const classNameBySize = {
    sm: 'h-6',
    md: 'h-8',
};
const unstable_hardcodeControlSize = {
    sm: 24,
    md: 32,
};

// atomic:component:range
// atomic:component:range:type.function
export function Range({
    id,
    testId,
    className,
    value: valueFromProps,
    max = 100,
    min = 0,
    onChange,
    disabled,
    size = 'md',
}: IRangeProps) {
    /**
     * Нормализуем значение
     */
    const value = useMemo(
        () => normalizeRangeValue(valueFromProps, min, max),
        [valueFromProps, min, max],
    );
    const rootRef = useRef<HTMLDivElement>(null);
    const space = max - min;
    const controlWidth = unstable_hardcodeControlSize[size];

    const leftPosition = 100 * ((value[0] - min) / space);
    const rightPosition = 100 * ((value[1] - min) / space);

    const getNextValue = (prev: number, diff: number, min: number, max: number) => {
        const width = rootRef.current?.clientWidth ?? 1;
        const next = prev + diff * (space / width);

        return Math.max(min, Math.min(max, next));
    };

    const minDrag = useDraggable({
        onMove({ diff: { x } }) {
            onChange?.([getNextValue(value[0], x, min, value[1]), value[1]]);
        },
    });
    const maxDrag = useDraggable({
        onMove({ diff: { x } }) {
            onChange?.([value[0], getNextValue(value[1], x, value[0], max)]);
        },
    });

    return (
        <div
            id={id}
            ref={rootRef}
            data-qa={testId}
            className={clsx('relative', classNameBySize[size], className)}
        >
            <div
                className={clsx(
                    'z-0 h-[3px] rounded-full bg-control-border',
                    'absolute left-0 top-1/2 right-0 -translate-y-1/2',
                )}
            />
            <div
                className={clsx(
                    'z-10 h-[3px] transition',
                    'absolute top-1/2 -translate-y-1/2',
                    disabled ? 'bg-text-disabled' : 'bg-text-primary',
                )}
                style={{
                    left: leftPosition + '%',
                    right: `${100 - rightPosition}%`,
                }}
            />
            <RangeControl
                className="absolute top-0 z-20"
                disabled={disabled}
                active={minDrag.dragging}
                size={size}
                min={min}
                max={max}
                value={value[0]}
                style={{
                    left: `calc(${leftPosition}%)`,
                }}
                {...minDrag.anchorProps}
            />
            <RangeControl
                className="absolute top-0 z-20"
                disabled={disabled}
                active={maxDrag.dragging}
                size={size}
                min={min}
                max={max}
                value={value[1]}
                style={{
                    left: `calc(${rightPosition}% - ${controlWidth}px)`,
                }}
                {...maxDrag.anchorProps}
            />
        </div>
    );
}
