import React, { ForwardedRef, forwardRef, useMemo, useRef, useState } from 'react';
import clsx from 'clsx';

import type { ISelectProps, SelectComponent, TSelectSize, TSelectValue } from './select.types';
import useId from '../../hooks/use-id';
import useEventCallback from '../../hooks/use-event-callback';
import Icon from '../../atoms/icon';
import ListBoxItem from '../../atoms/list-box-item';
import Popover from '../../molecules/popover';
import Button from '../../molecules/button';
import { useForkRef } from '../../hooks';

// atomic:component:select
// atomic:component:select:type.forward-ref
export const Select: SelectComponent = forwardRef(
    (
        {
            id: idProp,
            className,
            data,
            value,
            onChange,
            disabled,
            invalid,
            placeholder,
            size = 'md',
            theme = 'secondary',
            ...props
        }: ISelectProps<TSelectSize>,
        ref: ForwardedRef<HTMLButtonElement>,
    ) => {
        const id = useId(idProp);
        const rootRef = useRef<HTMLButtonElement | null>(null);
        const rootForkRef = useForkRef(rootRef, ref);
        const [open, setOpen] = useState(false);
        const optionByValue = useMemo(
            () => Object.fromEntries(data.map(option => [option.value, option])),
            [data],
        );

        const expanded = !disabled && open;
        const selected = value ? optionByValue[value] : null;

        const handleClick = useEventCallback(() => setOpen(true));
        const handleKeyDown = useEventCallback((e: React.KeyboardEvent) => {
            if ((!disabled && e.key === 'ArrowDown') || e.key === 'Enter') {
                e.stopPropagation();
                e.preventDefault();
                setOpen(true);
            }
        });

        const handleSelect = useEventCallback((selectedValue: TSelectValue) => {
            if (!disabled && selectedValue !== selected?.value) {
                onChange?.(selectedValue);
                setOpen(false);
            }
        });

        return (
            <>
                <Button
                    theme={theme}
                    size={size}
                    textSize={textSizes[size]}
                    disabled={disabled}
                    role="combobox"
                    aria-haspopup="listbox"
                    aria-expanded={expanded}
                    aria-controls={id}
                    aria-invalid={invalid}
                    aria-owns={`${id}-items`}
                    ref={rootForkRef as any}
                    onKeyDown={handleKeyDown}
                    onClick={handleClick}
                    className={clsx(
                        'w-full justify-between',
                        theme !== 'outlined' && 'border border-transparent',
                        invalid && 'border-solid border-red outline-red',
                        className,
                        disabled ? 'text-grey-light' : 'text-grey-dark',
                    )}
                    endNode={
                        <Icon
                            group={size === 'sm' ? '12' : '16'}
                            name="ArrowBottom"
                            className={clsx(
                                'flex items-center justify-center',
                                'rounded-full cursor-pointer bg-transparent text-gray-400',
                                'transition-colors hover:bg-gray-100 active:bg-gray-200',
                                !expanded && 'scale-100',
                            )}
                        />
                    }
                    {...props}
                >
                    {selected?.label ?? placeholder}
                </Button>
                <Popover
                    id={`${id}-items`}
                    aria-labelledby={`${id}-label`}
                    role="listbox"
                    open={expanded}
                    anchorNode={rootRef.current}
                    onClose={() => setOpen(false)}
                    anchorOrigin={{
                        horizontal: 'start',
                        vertical: 'end',
                    }}
                    transformOrigin={{
                        horizontal: 'start',
                        vertical: 'start',
                    }}
                    className="mt-1 py-1 bg-white"
                    style={
                        rootRef.current
                            ? {
                                  minWidth: rootRef.current.clientWidth,
                              }
                            : {}
                    }
                >
                    {data.map(item => (
                        <ListBoxItem
                            key={item.value}
                            onClick={() => handleSelect(item.value)}
                            selected={item === selected}
                            data-autofocus={item === selected}
                            reserveStartNodeSpace
                            startNode={item === selected ? <Icon name="Included" /> : null}
                        >
                            {item.label}
                        </ListBoxItem>
                    ))}
                </Popover>
            </>
        );
    },
) as any;

const textSizes = {
    sm: 'sm',
    md: 'sm',
    lg: 'md',
} as const;
