import {
    ComponentPropsWithoutRef,
    ComponentPropsWithRef,
    ElementType,
    JSXElementConstructor,
    RefAttributes,
} from 'react';

export type Nil = null | void | undefined;
export type HTMLElementType = keyof JSX.IntrinsicElements;

/**
 * @example RefOf<'div'> => Ref<HTMLDivElement | null>
 */
export type RefOf<C> = C extends ElementType ? ComponentPropsWithRef<C>['ref'] : never;

/**
 * @example PropsOf<'button'>
 */
export type PropsOf<C extends HTMLElementType | JSXElementConstructor<any>> =
    JSX.LibraryManagedAttributes<C, ComponentPropsWithoutRef<C>>;

/**
 * @example ExtendedProps<{ name: string; value: number }, { name: number }> => { name: number; value: number; }
 */
type ExtendedProps<_ExtendedProps = {}, OverrideProps = {}> = OverrideProps &
    Omit<_ExtendedProps, keyof OverrideProps>;

/**
 * @example InheritedProps<'div', { name: number }> => { name: number; ...props of div }
 */
type InheritedProps<C extends ElementType, Props = {}> = ExtendedProps<PropsOf<C>, Props>;

interface PolymorphicTagProps<C extends ElementType> {
    as?: C;
}

/**
 * @example PolymorphicProps<'button', { size?: 'large' }> => { size?: 'large', ...props of button }
 */
export type PolymorphicProps<C, Props = {}> = C extends ElementType
    ? InheritedProps<C, Props & PolymorphicTagProps<C>> & RefAttributes<RefOf<C>>
    : Props & PolymorphicTagProps<ElementType>;
