import { Logger } from 'logger';

import { BlackBoxClientOptions } from './types/client';
import {
    BLACKBOX_ALIASES_MAP,
    BLACKBOX_ATTRIBUTES_MAP,
    BLACKBOX_EMAIL_ATTRIBUTES_MAP,
    BLACKBOX_PHONE_ATTRIBUTES_MAP,
    BlackBoxCommonParams,
    BlackBoxSessionIdParams,
    BlackBoxSessionIdResponse,
} from './types';

import { TvmClient } from '../tvm';
import { compactObject, YandexCookieName, YandexHeaderName } from '../shared';
import { BlackBoxError, BlackBoxErrorCode, BlackBoxExceptionError } from './error';

export class BlackBoxClient<TvmDependencies extends string> {
    static defaultLogger = Logger.default.child({
        name: 'BlackBox',
    });

    readonly tvm: TvmClient<'blackbox' | TvmDependencies> | null = null;
    private readonly logger: Logger;
    private readonly baseUrl: URL;
    private readonly defaults: Record<string, any>;

    constructor({
        url,
        tvm,
        logger = BlackBoxClient.defaultLogger,
        defaults = {},
    }: BlackBoxClientOptions<TvmDependencies>) {
        this.baseUrl = new URL('/blackbox', url);
        this.tvm = tvm ?? null;
        this.logger = logger;
        this.defaults = this.prepareParams(defaults);
    }

    sessionid({
        host,
        userip,
        [YandexCookieName.SESSION_ID]: sessionid,
        [YandexCookieName.SESSION_ID_SSL]: sslsessionid,
        [YandexHeaderName.X_YA_SERVICE_TICKET]: serviceTicket,
        get_user_ticket = true,

        ...params
    }: BlackBoxSessionIdParams): Promise<BlackBoxSessionIdResponse> {
        return this.request('sessionid', {
            host,
            userip,
            sessionid,
            sslsessionid,
            get_user_ticket,
            [YandexHeaderName.X_YA_SERVICE_TICKET]: serviceTicket,
            ...this.defaults,
            ...this.prepareParams(params),
        });
    }

    private async request<
        T extends Partial<{ [YandexHeaderName.X_YA_SERVICE_TICKET]: string } & Record<string, any>>,
    >(method: string, { [YandexHeaderName.X_YA_SERVICE_TICKET]: ticket, ...query }: T) {
        const url = new URL(this.baseUrl);

        this.logger.debug(
            {
                query,
            },
            'Request %s',
            method,
        );

        for (const [name, value] of Object.entries({
            method,
            ...query,
        })) {
            url.searchParams.append(name, value as string);
        }

        const response = await fetch(url.toString(), {
            headers: {
                [YandexHeaderName.X_YA_SERVICE_TICKET]: await this.getServiceTicket(ticket),
            },
            method: 'get',
        });

        if (!response.ok) {
            throw new BlackBoxError(
                `Invalid response (${response.status} ${
                    response.statusText
                } - "${await response.text()}")`,
                BlackBoxErrorCode.INVALID_REQUEST,
            );
        }

        const json = await response.json();

        BlackBoxExceptionError.assert(json);

        return json;
    }

    private async getServiceTicket(value?: string): Promise<string> {
        if (!value) {
            if (!this.tvm) {
                throw new BlackBoxError(
                    'No X_YA_SERVICE_TICKET or tvm client',
                    BlackBoxErrorCode.MISSED_SERVICE_TICKET,
                );
            }

            return this.tvm.getTickets().then(tickets => tickets.asRecord.blackbox.ticket);
        }

        return value;
    }

    private prepareParams<
        T extends Omit<BlackBoxCommonParams, YandexHeaderName.X_YA_SERVICE_TICKET>,
    >({
        dbfields,
        aliases,
        attributes,
        email_attributes,
        phone_attributes,
        ...params
    }: T): Record<string, any> {
        return compactObject({
            ...params,
            format: 'json',
            dbfields:
                dbfields &&
                (Array.isArray(dbfields) ? dbfields.join(',') : Object.values(dbfields).join(',')),
            aliases: prepareAttributes(BLACKBOX_ALIASES_MAP, aliases),
            attributes: prepareAttributes(BLACKBOX_ATTRIBUTES_MAP, attributes),
            email_attributes:
                params.getemails &&
                prepareAttributes(BLACKBOX_EMAIL_ATTRIBUTES_MAP, email_attributes),
            phone_attributes:
                params.getphones &&
                prepareAttributes(BLACKBOX_PHONE_ATTRIBUTES_MAP, phone_attributes),
        });
    }
}

const prepareAttributes = <Name extends string, Map extends Record<Name, string | number>>(
    map: Map,
    value?: (Name | Map[Name])[],
) => value?.map(item => map[item as Name] ?? item.toString()).join(',');
