/* eslint-disable @typescript-eslint/naming-convention */

import { BLACKBOX_ATTRIBUTES_MAP } from './common-attributes';
import { YandexHeaderName } from '../../shared';

export interface BlackBoxAuthorizationStatus {
    /**
     * Number representation of authorization status:
     * 0 - account exists and active
     * 1 – need to update session cookies
     * 5 – account blocked
     * 6 – account doesn't exist or need to change password
     *
     * See:
     * https://nda.ya.ru/3UNKcC
     */
    id: number;
    /**
     * Статус авторизации (результат проверки авторизационной куки Session_id).
     *
     * Если в запросе был задан параметр multisession, элемент отражает общий статус куки.
     * В противном случае — статус куки, совмещенный со статусом аккаунта по умолчанию.
     *
     * Возможные статусы, с соответствующими числовыми идентификаторами:
     * - 0 (VALID) — переданная авторизационная кука валидна. Если не задан параметр multisession,
     *      этот статус также говорит о том, что аккаунт по умолчанию залогинен.
     * - 1 (NEED_RESET) — аутентификационная кука валидна, но может быть обновлена.
     *      Процесс обновления куки описан в документе Обзор системы авторизации.
     * - 2 (EXPIRED) — срок действия авторизационнойкуки истек, сессия не действительна.
     * - 3 (NOAUTH) — пользователь не авторизован в данном домене.
     *      Куки со статусом NOAUTH используются для подчиненных доменов в мультидоменной авторизации (МДА).
     * - 4 (DISABLED) — учетная запись пользователя существует, но заблокирована службой поддержки
     *      (возможен только если в запросе не был задан параметр multisession).
     * - 5 (INVALID) — авторизационная кука неправильного формата, была сформирована до выхода
     *      или проверяется не в том окружении, для которого была выставлена.
     *      Если в запросе не был задан параметр multisession, статус также может означать,
     *      что для аккаута был выполнен выход на всех устройствах, необходимо сменить пароль, или аккаунта не существует.
     */
    value: 'VALID' | 'NEED_RESET' | 'DISABLED' | 'INVALID' | 'EXPIRED' | 'NOAUTH';
}

/**
 * @see https://docs.yandex-team.ru/blackbox/methods/sessionid#standartnye-dopolnitelnye-argumenty
 */
export interface BlackBoxCommonParams {
    /**
     * Если Service-Ticket не был передан - будет использован TvmClient.
     * Если и его нет - ошибка
     */
    [YandexHeaderName.X_YA_SERVICE_TICKET]?: string;
    /**
     * Признак запроса алиасов аккаунта.
     * @example ['portal', 'phonenumber']
     * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#aliases
     */
    aliases?: BlackBoxAlias<BlackBoxAliasName>[];
    attributes?: BlackBoxAttribute<BlackBoxAttributeName>[];
    /**
     * Список имен полей в терминах старой базы Паспорта.
     *
     * Это устаревший способ получения данных об аккаунте,
     * информацию из паспортной базы данных рекомендуется запрашивать с помощью параметра attributes.
     * Значения поддерживаемых полей старой базы вычисляются на основе атрибутов, неизвестные поля игнорируются.
     * Некоторые поля требуют грантов для чтения, это отдельные гранты, независимые от грантов на чтение атрибутов.
     *
     * @deprecated
     */
    dbfields?: string[] | Record<string, string>;
    /**
     * Признак того, что в ответе следует вернуть дополнительные данные: имя пользователя для отображения в интерфейсе,
     * аватар и социальный профиль. Дополнительные данные появятся в блоке display_name.
     *
     * Для получения имени гранты не требуются.
     */
    regname?: boolean;
    /**
     * Признак наличия в ответе публичного имени пользователя. Данный параметр обрабатывается только совместно
     * с параметром regname, публичное имя добавляется внутри блока display_name.
     *
     * Поскольку публичное имя может содержать персональные данные, параметр требует наличия гранта allow_get_public_name.
     * @see https://docs.yandex-team.ru/blackbox/concepts/ugc#public-name
     */
    get_public_name?: boolean;
    /**
     * Признак наличия в ответе публичного идентификатора пользователя. Грантов не требует.
     * @see https://docs.yandex-team.ru/blackbox/concepts/ugc#public-id
     */
    get_public_id?: boolean;
    /**
     * Признак того, что в ответе следует вернуть список электронных адресов пользователя.
     * Набор адресов в ответе возвращается в элементе address-list и зависит от значения аргумента:
     * getall — возвращается список всех известных адресов пользователя, как внутренних, так и внешних.
     * getyandex — возвращается список адресов пользователя на Яндексе
     * getdefault — возвращается e-mail по умолчанию для данного пользователя.
     * testone — возвращается адрес, указанный в значении аргумента addrtotest, если этот адрес принадлежит учетной записи.
     *      Если указанный адрес не принадлежит пользователю, возвращается пустой список.
     *
     * @see https://docs.yandex-team.ru/blackbox/concepts/emails
     */
    emails?: 'getall' | 'getyandex' | 'getdefault' | 'testone';
    /**
     * E-mail, который проверяется на принадлежность пользователю.
     *
     * Аргумент обязателен, если в запросе присутствует аргумент emails со значением «testone».
     * Игнорируется во всех остальных случаях.
     */
    addrtotest?: boolean;
    /**
     * Признак того, что в ответе следует вернуть список атрибутов внешних адресов пользователя.
     * Возможные значения:
     * all — возвращает атрибуты для всех известных почтовых адресов.
     *
     * Номера почтовых атрибутов перечисляются в аргументе email_attributes.
     * Если аргумент email_attributes не передан, в ответе появится только список идентификаторов e-mail-ов.
     */
    getemails?: 'all';
    /**
     * Список номеров почтовых атрибутов, которые следует вернуть в ответе.
     * Полный список атрибутов приведен в таблице атрибутов e-mail-ов.
     * Аргумент необходимо передавать совместно с аргументом getemails.
     *
     * Гранты на получение атрибутов привязанных e-mail-ов не требуются.
     */
    email_attributes?: BlackBoxEmailAttribute<BlackBoxEmailAttributeName>[];
    /**
     * Признак того, что в ответе следует вернуть список атрибутов привязанных телефонов пользователя.
     * Возможные значения:
     * all — возвращает все сохраненные в профиле телефоны.
     * bound — возвращает только привязанные (подтвержденные) телефоны.
     *
     * Номера телефонных атрибутов перечисляются в аргументе phone_attributes.
     * Если аргумент phone_attributes не передан, в ответе появится только список идентификаторов телефонов.
     * @see https://docs.yandex-team.ru/blackbox/concepts/phones
     * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#section_kbk_b5f_2hb
     */
    getphones?: 'all' | 'bound';
    /**
     * Список номеров телефонных атрибутов, которые следует вернуть в ответе.
     * Полный список атрибутов приведен в таблице атрибутов телефонов.
     * Аргумент необходимо передавать совместно с аргументом getphones.
     *
     * Атрибуты, содержащие полный телефонный номер, являются персональными данными и требуют гранта для их чтения.
     * @see https://docs.yandex-team.ru/blackbox/concepts/phones
     * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#section_kbk_b5f_2hb
     */
    phone_attributes?: BlackBoxPhoneAttribute<BlackBoxPhoneAttributeName>[];
}

export type BlackBoxEmailAttribute<Name extends BlackBoxEmailAttributeName> =
    | Name
    | BlackBoxEmailAttributesMap[Name];
export type BlackBoxEmailAttributeName = keyof BlackBoxEmailAttributesMap;

export type BlackBoxPhoneAttribute<Name extends BlackBoxPhoneAttributeName> =
    | Name
    | BlackBoxPhoneAttributesMap[Name];
export type BlackBoxPhoneAttributeName = keyof BlackBoxPhoneAttributesMap;

export type BlackBoxAttribute<Name extends BlackBoxAttributeName> =
    | Name
    | BlackBoxAttributesMap[Name];
export type BlackBoxAttributeName = keyof BlackBoxAttributesMap;

export type BlackBoxAlias<Name extends BlackBoxAliasName> = Name | BlackBoxAliasesMap[Name];
export type BlackBoxAliasName = keyof BlackBoxAliasesMap;

/**
 * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#db-attributes
 */
export type BlackBoxAttributesMap = typeof BLACKBOX_ATTRIBUTES_MAP;

/**
 * Типы алиасов (конкретные данные, привязанные к учетке)
 * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#aliases
 * @see https://docs.yandex-team.ru/authdevguide/concepts/Account_About#section_lry_jnq_pcb
 */
export type BlackBoxAliasesMap = typeof BLACKBOX_ALIASES_MAP;

/**
 * Почтовые атрибуты
 * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#section_gyv_v5f_2hb
 */
export type BlackBoxEmailAttributesMap = typeof BLACKBOX_ALIASES_MAP;

/**
 * Телефонные атрибуты
 * @see https://docs.yandex-team.ru/authdevguide/concepts/DB_About#section_kbk_b5f_2hb
 */
export type BlackBoxPhoneAttributesMap = typeof BLACKBOX_ALIASES_MAP;

export const BLACKBOX_ALIASES_MAP = {
    /**
     * Алиас портального аккаунта. Может использоваться в любом контексте, где требуется логин пользователя.
     */
    portal: 1,
    /**
     * Почтовый алиас. Используется пользователями, у которых адрес почтового ящика не совпадает с портальным алиасом.
     * Используется при доставке почты или поиске по e-mail. Аккаунты с данным алиасом также обязательно имеют портальный алиас.
     */
    mail: 2,
    /**
     * Алиас почты на Народе, если он не совпадает с портальным алиасом.
     * Аккаунты с данным алиасом также обязательно имеют портальный алиас.
     */
    narodmail: 3,
    /**
     * Алиас страницы пользователя на Народе, если адрес не совпадает с портальным.
     * Аккаунты с данным алиасом также обязательно имеют портальный алиас.
     */
    narod: 4,
    /**
     * Лайт-пользователь. В алиасе хранится подтвержденный e-mail стороннего сервиса.
     */
    lite: 5,
    /**
     * Социальный аккаунт. В алиасе хранится сгенерированный идентификатор, недоступный пользователю.
     * Аккаунт может быть дорегистрирован и иметь портальный алиас.
     */
    social: 6,
    /**
     * Пользователь Почты Для Домена. Хранит информацию о пользователе в формате пользователь@домен.
     * Доменная часть идентификатора может меняться при редактировании имени домена.
     */
    pdd: 7,
    /**
     * Алиасы ПДД-пользователя. Пользователь ПДД может иметь несколько алиасов.
     */
    pddalias: 8,
    /**
     * Аккаунты в альтернативных доменах, принадлежащих Яндексу. Аналогичен алиасу ПДД.
     */
    altdomain: 9,
    /**
     * Алиас телефонного аккаунта. Не рекомендуется использовать из-за сложностей
     * с переходом телефонного номера от одного владельца другому.
     */
    phonish: 10,
    /**
     * Телефонный алиас. Обеспечивает поиск по телефонному номеру, уникальному для портального аккаунта.
     */
    phonenumber: 11,
    /**
     * Технический алиас для сбора почты со сторонних почтовых сервисов.
     */
    mailish: 12,
    /**
     * Логин сотрудника Яндекса, подтвердившего свой аккаунт на Стаффе. Наличие алиаса является признаком сотрудника.
     */
    yandexoid: 13,
    /**
     * Технический алиас, содержащий идентификатор аккаунта Кинопоиска.
     */
    kinopoisk: 15,
    /**
     * Технический алиас, содержащий идентификатор аккаунта Uber.
     */
    uberid: 16,
    /**
     * Технический алиас для идентификации ботов или каналов нового мессенджера.
     */
    yambot: 17,
    /**
     * Технический алиас для идентификации пользователей Яндекс.Станции.
     */
    kolonkish: 18,
} as const;

export const BLACKBOX_PHONE_ATTRIBUTES_MAP = {
    /**
     * Номер телефона. Требует гранта ЧЯ.
     */
    'phone.number': 1,
    /**
     * Время создания записи. Требует гранта ЧЯ.
     */
    'phone.created': 2,
    /**
     * Время привязки телефона к аккаунту. Требует гранта ЧЯ.
     */
    'phone.bound': 3,
    /**
     * Время подтверждения владения номером Требует гранта ЧЯ.
     */
    'phone.confirmed': 4,
    /**
     * Время подтверждения пользователем владения номером (варианты: approved, conceded, accepted, agreed). Требует гранта ЧЯ.
     */
    'phone.admitted': 5,
    /**
     * Время получения статуса защищённости номера Требует гранта ЧЯ.
     */
    'phone.secured': 6,
    /**
     * Номер в international формате (+7 999 111-22-33).Требует гранта ЧЯ.
     */
    'phone.formatted_number': 101,
    /**
     * Номер в e164 (+79991112233). Требует гранта ЧЯ.
     */
    'phone.e164_number': 102,
    /**
     * Форматированный маскированный номер (+7 999 ***‒**‒33).
     */
    'phone.masked_formatted_number': 103,
    /**
     * Маскированный номер в e164 (+7999*****33).
     */
    'phone.masked_e164_number': 104,
    /**
     * Булевский флаг: подтвержден ли номер. 1, если пользователь ввёл проверочный код, который был отправлен на данный телефон.
     */
    'phone.is_confirmed': 105,
    /**
     * Булевский флаг: привязан ли номер.1, если пользователь так или иначе связал данный телефон со своим аккаунтом (на данный момент привязанность влечёт за собой подтвержденность)
     */
    'phone.is_bound': 106,
    /**
     * Булевский флаг: дефолтный ли номер. 1, если данный телефон следует использовать для общения с пользователем.
     */
    'phone.is_default': 107,
    /**
     * Булевский флаг: защищен ли номер. 1, если данный телефон используется для восстановления доступа к аккаунту. Одновременно может быть только один защищенный номер.
     */
    'phone.is_secured': 108,
} as const;

export const BLACKBOX_EMAIL_ATTRIBUTES_MAP = {
    /**
     * Внешний адрес электронной почты
     */
    'email.address': 1,
    /**
     * Время создания email'а
     */
    'email.created': 2,
    /**
     * Время подтверждения email'а
     */
    'email.confirmed': 3,
    /**
     * Время привязки email'а
     */
    'email.bound': 4,
    /**
     * Сборщик писем
     */
    'email.is_rpop': 5,
    /**
     * Запрет использования почтового ящика для восстановления пароля
     */
    'email.is_unsafe': 6,
    /**
     * Запрет рассылок и нотификаций на email
     */
    'email.is_silent': 7,
} as const;
