import {
    ITvmAuthResult,
    ITvmHostParams,
    ITvmValidatedTicket,
    TvmAuthPolicy,
    TvmErrorCode,
} from './types';

import { TvmError } from './error';
import { YandexHeaderName } from '../shared';

// TODO Вынести в клиент

export function checkTvmAuthPolicy(
    policy: TvmAuthPolicy,
    allowed: number[],
    incoming?: ITvmValidatedTicket | null,
): ITvmAuthResult {
    const id = incoming?.src;

    if (id && allowed.includes(id)) {
        return {
            authorized: true,
            message: `Incoming TVM id ${id} successfully authenticated`,
        };
    }

    switch (policy) {
        case TvmAuthPolicy.OPTIONAL:
            return {
                authorized: false,
                message: id
                    ? 'Incoming TVM id is not found in allowed ids list, optional auth skipped'
                    : 'No incoming TVM id found, optional auth skipped',
            };
        case TvmAuthPolicy.REQUIRED:
            throw new TvmError(
                id
                    ? 'Incoming TVM id is not found in allowed ids list, auth failed'
                    : 'No incoming TVM id found, auth failed',
                TvmErrorCode.INCOMING_UNAUTHORIZED,
            );
        default:
            throw new TvmError(
                `Policy ${policy} is not implemented`,
                TvmErrorCode.INCOMING_UNKNOWN_AUTH_POLICY,
            );
    }
}

export async function validateIncomingTvmTicket(
    { url, token }: ITvmHostParams,
    ticket?: string | null,
) {
    if (!ticket) return null;

    const headers = new Headers({
        [YandexHeaderName.AUTHORIZATION]: token,
        [YandexHeaderName.X_YA_SERVICE_TICKET]: ticket,
    });
    const response = await fetch(new URL('/tvm/checksrv', url).toString(), {
        headers,
    });

    if (!response.ok) {
        throw new TvmError(response.statusText, TvmErrorCode.INCOMING_INVALID_TICKET);
    }

    const data = await response.json();

    return data as ITvmValidatedTicket;
}
