import { Logger } from 'logger';

import { ITvmTicket, TTvmResult, TvmClientOptions, TvmErrorCode, TvmTicketsResult } from './types';

import { YandexHeaderName } from '../shared';
import { getTvmErrorFromFailedResponse, TvmError } from './error';

export class TvmClient<DependencyName extends string> {
    static defaultLogger = Logger.default.child({
        name: 'TVM',
    });

    private readonly logger: Logger;
    private readonly baseUrl: URL;
    private readonly baseHeaders: Headers;
    private readonly dependenciesIds: number[];
    private readonly dependenciesNames: DependencyName[];
    private readonly cacheTime: number | false;
    private cache: ITvmCache<DependencyName> | null = null;

    constructor({
        dependencies,
        logger = TvmClient.defaultLogger,
        token,
        url,
        cacheTime = 1000 * 60,
    }: Readonly<TvmClientOptions<DependencyName>>) {
        this.baseUrl = new URL(url);
        this.baseUrl.searchParams.append('dsts', dependencies.map(dep => dep.id).join(','));

        this.baseHeaders = new Headers({
            [YandexHeaderName.AUTHORIZATION]: token,
        });
        this.dependenciesIds = dependencies.map(dep => Number(dep.id));
        this.dependenciesNames = dependencies.map(dep => dep.name);
        this.cacheTime = cacheTime;
        this.logger = logger;
    }

    getTickets() {
        if (!this.cacheTime) return this.getActualTickets();

        const time = Date.now();

        if (!this.cache || this.cache.expiresIn < time) {
            this.cache = {
                expiresIn: time + this.cacheTime,
                promise: this.getActualTickets().catch(error => {
                    this.cache = null;
                    throw error;
                }),
            };
        }

        return this.cache.promise;
    }

    async getActualTickets(): Promise<TvmTicketsResult<DependencyName>> {
        const url = new URL('/tvm/tickets', this.baseUrl);

        url.searchParams.append('dsts', this.dependenciesIds.join(','));

        this.logger.debug(
            {
                destinations: this.dependenciesIds,
            },
            'Get tickets (%s)',
            url.toString(),
        );

        const response = await fetch(url.toString(), {
            headers: this.baseHeaders,
            method: 'get',
        });

        if (!response.ok) {
            const error = await getTvmErrorFromFailedResponse(response, url.toString());

            this.logger.debug({ err: error }, 'Tickets request failed');
            throw error;
        }

        const tickets: Record<DependencyName, ITvmTicket> = await response.json();
        const missed = this.dependenciesNames.filter(name => !tickets[name]?.ticket);
        const errors = Object.values<ITvmTicket>(tickets)
            .map(ticket => ticket.error)
            .filter(Boolean);

        if (errors.length) {
            throw new TvmError(
                `[TVM] Tickets resolved with errors: ${errors.join(', ')}`,
                TvmErrorCode.TICKETS_ERROR,
            );
        }

        if (missed.length) {
            throw new TvmError(
                `[TVM] Missed tickets: ${missed.join(', ')}`,
                TvmErrorCode.TICKETS_ERROR,
            );
        }

        const asList = this.dependenciesNames.map(name => ({
            id: tickets[name].tvm_id,
            name,
            ticket: tickets[name].ticket!,
            original: tickets[name],
        }));

        this.logger.debug({ tickets: asList }, 'Tickets resolved');

        return {
            asList,
            asRecord: Object.fromEntries(
                asList.map(ticket => [ticket.name, ticket]),
            ) as TTvmResult<DependencyName>,
        };
    }
}

interface ITvmCache<DependencyName extends string> {
    expiresIn: number;
    promise: Promise<TvmTicketsResult<DependencyName>>;
}
