/* eslint-disable @typescript-eslint/naming-convention */

import { Logger } from 'logger';

import { URLInit } from '../shared';

export interface TvmClientOptions<Dependencies extends string> {
    logger?: Logger;
    /**
     * Максимальное время сохранения токенов в памяти
     * @default 60000 (1 минута)
     */
    cacheTime?: number | false;
    /**
     * Список ID требуемых сервисов
     * @example [{ name: 'blackbox', id: 224 }, { name: 'foo-bar', id: 2032429 }]
     */
    dependencies: ITvmDependency<Dependencies>[];
    /**
     * Токен, записанный в TVM-конфиге
     * @example 0801111123111111907611111883111
     */
    token: string;
    /**
     * URL TVM-демона
     * @example http://localhost:9011
     */
    url: URLInit;
}

export enum TvmAuthPolicy {
    DISABLED = 'disabled',
    OPTIONAL = 'optional',
    REDIRECT = 'redirect',
    REQUIRED = 'required',
}

// eslint-disable-next-line @typescript-eslint/naming-convention
export enum TvmErrorCode {
    INCOMING_INVALID_TICKET = 'INCOMING_INVALID_TICKET',
    INCOMING_UNAUTHORIZED = 'INCOMING_UNAUTHORIZED',
    INCOMING_UNKNOWN_AUTH_POLICY = 'INCOMING_UNKNOWN_AUTH_POLICY',
    INVALID_REQUEST = 'TVM_INVALID_REQUEST',
    INVALID_TOKEN = 'TVM_INVALID_TOKEN',
    REQUEST_ERROR = 'TVM_REQUEST_ERROR',
    TICKETS_ERROR = 'TVM_TICKETS_ERROR',
}

export interface ITvmAuthResult {
    authorized: boolean;
    message: string;
}

export interface ITvmValidatedTicket {
    src: number;
    dst: number;
}

export interface ITvmParams<DependencyName extends string> extends ITvmHostParams {
    /**
     * Список ID требуемых сервисов
     * @example [{ name: 'blackbox', id: 224 }, { name: 'foo-bar', id: 2032429 }]
     */
    dependencies: ITvmDependency<DependencyName>[];
    // Возможно, не нужны
    headers?: Headers;
}

export interface ITvmHostParams {
    /**
     * Токен, записанный в TVM-конфиге
     * @example 0801111123111111907611111883111
     */
    token: string;
    /**
     * URL TVM-демона
     * @example http://localhost:9011
     */
    url: string | URL;
}

/**
 * Тикет, который возвращает tvm-демон
 * @see https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/#/v2/tickets
 */
export interface ITvmTicket {
    /** An error which occurred while getting ticket */
    readonly error?: string;

    /** TVM service ticket */
    readonly ticket?: string;

    /** TVM service id */
    readonly tvm_id: number;
}

export interface TvmTicketsResult<Name extends string> {
    asList: ITvmResolvedTicket<Name>[];
    asRecord: TTvmResult<Name>;
}

/**
 * Обработанный тикет после проверки на наличие ошибки
 */
export interface ITvmResolvedTicket<Name extends string> {
    readonly id: number;
    readonly name: Name;
    readonly ticket: string;
    readonly original: ITvmTicket;
}

export interface ITvmDependency<Name extends string> {
    id: string | number;
    name: Name;
}

export type TTvmResult<DependencyName extends string> = {
    [Name in DependencyName]: ITvmResolvedTicket<Name>;
};
