import { CookieSerializeOptions } from 'cookie';
import dayjs from 'dayjs';

import { IYandexUidParams, IYandexUidSupportedParams } from './types';

import { hasSameSiteSupport } from '../uatraits';

/**
 * Утилиты по работе с yandexuid
 * @see https://wiki.yandex-team.ru/Cookies/yandexuid
 * @see @yandex-int/express-yandexuid
 */

export const isYandexUidSupported = ({
    uatraits: { isBrowser, isRobot },
    domain,
    method,
}: IYandexUidSupportedParams) =>
    isBrowser && !isRobot && domain && isYandexUidSupportedDomain(domain) && method === 'GET';

export const createYandexUid = () =>
    (Math.random() * 1e9).toFixed() + String(Date.now()).slice(0, 10);

export const isValidYandexUid = (value: string) =>
    value &&
    !value.startsWith('0') &&
    YANDEX_UID_RE.test(value) &&
    Number.parseInt(value.slice(-10), 10) < Date.now() / 1000;

export const isYandexUidSupportedDomain = (domain: string) => YANDEX_DOMAINS.includes(domain);

// TODO Вынести в shared?
export const getYandexUidDomain = (host: string) => DOMAIN_RE.exec(host)?.[0] ?? null;

export function createYandexUidCookieParams({
    uatraits,
    domain,
    expires = dayjs().add(10, 'years').toDate(),
}: IYandexUidParams): CookieSerializeOptions {
    const sameSiteAvailable = hasSameSiteSupport(uatraits);

    return {
        domain: '.' + domain,
        path: '/',
        expires,
        secure: sameSiteAvailable,
        sameSite: sameSiteAvailable ? 'none' : false,
    };
}

// RegExp for checking the validity of an yandexuid string. 17-20 digits and starts with 0
const YANDEX_UID_RE = /^\d{17,20}$/;

// RegExp for extracting the second and the top level domains
const DOMAIN_RE =
    /[a-z-]+\.(az|by|co\.il|com|com\.am|com\.ge|com\.tr|ee|fr|kg|kz|lt|lv|md|ru|tj|tm|ua|uz|rs|cn)$/;

// List of yandex domains
// @see https://wiki.yandex-team.ru/Cookies/yandexuid#kudastavitsja
const YANDEX_DOMAINS = [
    'ya.ru',
    'yandex.az',
    'yandex.by',
    'yandex.co.il',
    'yandex.com',
    'yandex.com.am',
    'yandex.com.ge',
    'yandex.com.tr',
    'yandex.ee',
    'yandex.fr',
    'yandex.kg',
    'yandex.kz',
    'yandex.lt',
    'yandex.lv',
    'yandex.md',
    'yandex.ru',
    'yandex.tj',
    'yandex.tm',
    'yandex.ua',
    'yandex.uz',
    'yandex.rs',
    'yandex-ad.cn',
];
