import { NextApiRequest, NextApiResponse } from 'next';
import httpProxy, { ServerOptions } from 'http-proxy';

export interface INextApiHttpProxyOptions extends ServerOptions {
    pathRewrite?: Record<string, string> | INextApiHttpRewriteRule[];
}

export interface INextApiHttpRewriteRule {
    pattern: RegExp;
    replace: string;
}

export function proxyNextHttpRequest(
    req: NextApiRequest,
    res: NextApiResponse,
    { pathRewrite, ...options }: INextApiHttpProxyOptions,
) {
    const haveBody = isReqWithBody(req);
    const proxy = httpProxy.createProxy({
        changeOrigin: true,
        ...options,
    });

    if (haveBody && typeof req.body === 'object') {
        req.body = JSON.stringify(req.body);
    }

    if (pathRewrite) {
        req.url = rewriteProxyPath(req.url as string, prepareRewriteRules(pathRewrite));
    }

    return new Promise((resolve, reject) => {
        proxy
            .once('proxyReq', (proxyReq: any, req: any): void => {
                if (haveBody && typeof req.body === 'string') {
                    proxyReq.write(req.body);
                    proxyReq.end();
                }
            })
            .once('proxyRes', resolve)
            .once('error', reject)
            .web(req, res);
    });
}

export const rewriteProxyPath = (url: string, rules: INextApiHttpRewriteRule[]) => {
    const rule = rules.find(item => item.pattern.test(url));

    return rule ? url.replace(rule.pattern, rule.replace) : url;
};

const prepareRewriteRules = (
    input: INextApiHttpProxyOptions['pathRewrite'],
): INextApiHttpRewriteRule[] =>
    Array.isArray(input)
        ? input
        : Object.entries(input as Record<string, string>).map(([pattern, replace]) => ({
              pattern: new RegExp(pattern),
              replace,
          }));

/**
 * Please refer to the following links for the specification document for HTTP.
 * @see https://tools.ietf.org/html/rfc7231
 * @see https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol
 */
const METHODS_WITH_BODY = ['HEAD', 'POST', 'PUT', 'DELETE', 'CONNECT', 'OPTIONS', 'PATCH'];
const isReqWithBody = (req: NextApiRequest) => METHODS_WITH_BODY.includes(req.method as string);
