import { resolve } from 'path';
import glob from 'glob';
import { arc } from './arc-client.mjs';
import { workspace } from './workspace-client.mjs';
import { relative } from 'node:path';

export async function getChangesWithInfo({ staged, head, base }) {
    const root = await workspace.getRoot();
    const packages = await getWorkspacePackagesPaths(root);
    const workspaceArcPath = relative(arc.getRoot(), root);
    const changedNames = arc.getDiff({
        staged,
        cached: staged,
        base,
        head,
        nameOnly: true,
        status: true,
    });

    const changesPaths = changedNames
        .map(({ path }) => relative(workspaceArcPath, path))
        .filter(path => !path.startsWith('..'));

    const changedPackagesPaths = packages
        .map(path => ({
            path,
            changes: changesPaths.filter(changedPath => changedPath.startsWith(path)),
        }))
        .filter(item => item.changes.length > 0);

    const changedPackages = await Promise.all(
        changedPackagesPaths.map(async ({ path, changes }) => {
            const { default: packageJson } = await import(resolve(root, path, 'package.json'), {
                assert: { type: 'json' },
            });

            return {
                name: packageJson.name,
                path,
                absPath: resolve(root, path),
                changes,
                packageJson,
            };
        }),
    );

    return {
        changes: {
            files: changesPaths,
            packages: changedPackages,
        },
    };
}

export async function getWorkspacePackagesPaths(root) {
    const {
        default: {
            workspaces: { packages },
        },
    } = await import(`${root}/package.json`, {
        assert: { type: 'json' },
    });

    return await Promise.all(
        packages.map(
            pattern =>
                new Promise((resolve, reject) => {
                    glob(
                        pattern,
                        {
                            cwd: root,
                            root,
                        },
                        (error, result) => (error ? reject(error) : resolve(result)),
                    );
                }),
        ),
    ).then(paths => [].concat(...paths));
}
