#include "boiler.h"
#include "sender.h"

#include <travel/hotels/lib/cpp/util/sizes.h>
#include <library/cpp/logger/global/global.h>

#include <util/generic/guid.h>
#include <util/system/hostname.h>

#include <math.h>

namespace NTravel {
namespace NBoiler {

void ShiftKeyDates(TKey& key) {
    key.DateIn++;
    key.DateOut++;
}

void TBoiler::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NKeysInexistent));
    ct->insert(MAKE_COUNTER_PAIR(NKeysInexistentGreylisted));
    ct->insert(MAKE_COUNTER_PAIR(NKeysInexistentBoiled));
    ct->insert(MAKE_COUNTER_PAIR(NKeysExistent));
    ct->insert(MAKE_COUNTER_PAIR(NKeysExistentGreylisted));
    ct->insert(MAKE_COUNTER_PAIR(NKeysExistentError));

    ct->insert(MAKE_COUNTER_PAIR(NKeysShiftable));
    ct->insert(MAKE_COUNTER_PAIR(NKeysUnshiftable));

    auto now = Now();
    double existentUsages = ExistentUsages.GetCastedTo(now);
    double totalUsages = TotalUsages.GetCastedTo(now);
    BoiledMass = (fabs(totalUsages) < 1e-10) ? 0 : (existentUsages / totalUsages * 100);
    ct->insert(MAKE_COUNTER_PAIR(BoiledMass));

    ct->insert(MAKE_COUNTER_PAIR(NKeysShiftedToNew));
    ct->insert(MAKE_COUNTER_PAIR(NKeysShiftedToExisting));

    ct->insert(MAKE_COUNTER_PAIR(NBoils));
    ct->insert(MAKE_COUNTER_PAIR(NKeysObsolete));

    ct->insert(MAKE_COUNTER_PAIR(NBytesAllKeys));
    ct->insert(MAKE_COUNTER_PAIR(NBytesAllKeysForInvalidation));
    ct->insert(MAKE_COUNTER_PAIR(NBytesKeyExpirationSchedule));
    ct->insert(MAKE_COUNTER_PAIR(NBytesBoilableKeys));
    ct->insert(MAKE_COUNTER_PAIR(NBytes));
}

TBoiler::TBoiler(const NTravelProto::NBoiler::TConfig::TBoiler& cfg, TSender& sender, TGreylist& greylist, const TCacheInvalidationService& cacheInvalidationService)
    : Config_(cfg)
    , KeyBoilExpireTimeout_(TDuration::Seconds(cfg.GetKeyBoilExpireTimeoutSec()))
    , KeyAgeToGC_(TDuration::Seconds(cfg.GetKeyAgeToGCSec()))
    , Greylist_(greylist)
    , Counters_({"partner"})
    , Sender_(sender)
    , TillKeyGC_(cfg.GetKeyGCPeriodSec())
    , TillKeysExpiration_(cfg.GetKeysExpirationPeriodSec())
    , NeedShiftKeys_(false)
    , CacheInvalidationService_(cacheInvalidationService)
{
    UpdateDateToShift(Now());
    UpdateMinDate(Now());
}

TBoiler::~TBoiler() {
    Stop();
}

void TBoiler::RegisterCounters(NMonitor::TCounterSource& counters) {
    counters.RegisterSource(&Counters_, "Boiler");
}

void TBoiler::Start() {
    Thread_ = SystemThreadFactory()->Run(this);
}

void TBoiler::Stop() {
    if (Thread_) {
        StopFlag_.Set();
        WakeUp_.Signal();
        Thread_->Join();
        Thread_.Destroy();
    }
}

bool TBoiler::IsReady() const {
    return InitialBusReadDone_ && PartnersConfigGot_;
}

void TBoiler::OnInitialBusReadDone() {
    PartnerBoilers_.ForEach([](TPartnerBoilerRef b){ b->OnInitialBusReadDone();});
    InitialBusReadDone_.Set();
}

void TBoiler::OnUpdatePartnersConfig(const THashMap<EPartnerId, NTravelProto::NConfig::TPartner>& partnersConfig) {
    PartnerBoilers_.ForEach([](TPartnerBoilerRef partnerBoiler){
        partnerBoiler->SetRPS(0);
    });
    for (auto pcIt = partnersConfig.begin(); pcIt != partnersConfig.end(); ++pcIt) {
        GetOrCreatePartnerBoiler(pcIt->first)->SetRPS(pcIt->second.GetBoilerRPS());
    }
    PartnersConfigGot_.Set();
}

void TBoiler::OnRequestFinished(const NTravelProto::NBoiler::TSearchOffersShortReq& req, TInstant ts, TInstant expireTimestamp, bool error) {
    GetOrCreatePartnerBoiler(req.GetHotelId().GetPartnerId())->OnRequestFinished(req, ts, expireTimestamp, error);
}

void TBoiler::OnCacheKeyUse(const ru::yandex::travel::hotels::TOfferCacheReq& message, TInstant ts) {
    double usage;
    if (message.GetIsDefaultSubKey()) {
        usage = Config_.GetCostDefaultSubKey();
    } else if (message.GetIsSimilar()) {
        usage = Config_.GetCostSimilar();
    } else if (message.GetIsUseSearcher()) {
        usage = Config_.GetCostUseSearcher();
    } else if (message.GetIsFull()) {
        usage = Config_.GetCostFull();
    } else {
        usage = Config_.GetCostSimple();
    }
    TKey key;
    key.Currency = message.GetCurrency();
    key.DateIn = NOrdinalDate::FromString(message.GetCheckInDate());
    key.DateOut = NOrdinalDate::FromString(message.GetCheckOutDate());
    key.Occupancy = TAges::FromOccupancyString(message.GetOccupancy());
    bool shifted = false;
    {
        TReadGuard g(Lock_);
        if (key.DateIn < MinDate_) {
            return;
        }
        if (message.GetAllowDateShift() && key.DateIn == DateToShift_) {
            shifted = true;
            ShiftKeyDates(key);
        }
    }
    for (const auto& pbHotelId: message.GetHotelIdsWithPermalink()) {
        key.HotelId.PartnerId = pbHotelId.GetHotelId().GetPartnerId();
        key.HotelId.OriginalId = pbHotelId.GetHotelId().GetOriginalId();
        GetOrCreatePartnerBoiler(key.HotelId.PartnerId)->OnCacheKeyUse(message, key, pbHotelId.GetPermalink(), ts, usage, shifted);
    }
}

TBoiler::TPartnerBoilerRef TBoiler::GetOrCreatePartnerBoiler(EPartnerId pId) {
    return PartnerBoilers_.GetOrCreate(pId, [this, pId](){
       return new TPartnerBoiler(pId, *this);
    });
}

void TBoiler::DoExecute() {
    TInstant nextAction = Now();
    while (!StopFlag_) {
        nextAction += TDuration::Seconds(1);
        if (WakeUp_.WaitD(nextAction)) {
            continue;
        }
        UpdateDateToShift(nextAction);
        if (NeedShiftKeys_) {
            DoKeysShift();
        }
        UpdateMinDate(nextAction);
        if (--TillKeyGC_ == 0) {
            TillKeyGC_ = Config_.GetKeyGCPeriodSec();
            PartnerBoilers_.ForEach([nextAction](TPartnerBoilerRef b){ b->DoKeyGC(nextAction);});
        }
        if (IsReady()) {
            NOrdinalDate::TOrdinalDate minDate;
            {
                TReadGuard g(Lock_);
                minDate = MinDate_;
            }
            PartnerBoilers_.ForEach([nextAction, minDate](TPartnerBoilerRef b){ b->DoBoil(nextAction, minDate);});
        }
        if (--TillKeysExpiration_ == 0) {
            TillKeysExpiration_ = Config_.GetKeysExpirationPeriodSec();
            PartnerBoilers_.ForEach([nextAction](TPartnerBoilerRef b){ b->DoKeysExpiration(nextAction);});
        }
    }
}

TInstant TBoiler::GetUsageCastTs() const {
    TReadGuard g(Lock_);
    return UsageCastTs_;
}

void TBoiler::UpdateDateToShift(TInstant now) {
    auto dateToShift = NOrdinalDate::FromInstant(now + TDuration::Hours(Config_.GetHoursBeforeDateShift()));
    {
        TWriteGuard g(Lock_);
        if (dateToShift == DateToShift_) {
            return;
        }
        DateToShift_ = dateToShift;
    }
    NeedShiftKeys_ = true;
    INFO_LOG << "New date shift is " << dateToShift << Endl;
}

void TBoiler::DoKeysShift() {
    NOrdinalDate::TOrdinalDate dateToShift;
    {
        TReadGuard g(Lock_);
        dateToShift = DateToShift_;
    }

    INFO_LOG << "Start shift keys, shift date is " << dateToShift << Endl;
    NeedShiftKeys_ = false;
    size_t limit = Config_.GetMaxKeysToShiftAtOnce();
    PartnerBoilers_.ForEach([dateToShift, &limit](TPartnerBoilerRef b) { b->DoKeysShift(dateToShift, limit); });
    if (limit == 0) {
        NeedShiftKeys_ = true; // Будем продолжать потом!
        INFO_LOG << "Will continue shift next time" << Endl;
    } else {
        INFO_LOG << "Finished shift keys" << Endl;
    }
}

void TBoiler::UpdateMinDate(TInstant now) {
    NOrdinalDate::TOrdinalDate minDate = NOrdinalDate::FromInstant(now + TDuration::Hours(Config_.GetHoursBeforeDateObsolete()));
    {
        TWriteGuard g(Lock_);
        if (minDate == MinDate_) {
            return;
        }
        MinDate_ = minDate;
        UsageCastTs_ = NOrdinalDate::ToInstant(minDate);
        INFO_LOG << "New MinDate is " << MinDate_ << ", New UsageCastTs is " << UsageCastTs_ << Endl;
    }

    PartnerBoilers_.ForEach([](TPartnerBoilerRef b) {
        b->DoKeysRecast(false);
    });
}

TBoiler::TPartnerBoiler::TPartnerBoiler(EPartnerId partnerId, TBoiler& owner)
    : PartnerId(partnerId)
    , Owner(owner)
    , Counters(owner.Counters_.GetOrCreate({NTravelProto::EPartnerId_Name(partnerId)}))
    , InitialBusReadDone(owner.InitialBusReadDone_)
    , RPS(0)
    , AccumulatedRPS(0)
    , UsageCastTs(TInstant::Zero())
{
}

void TBoiler::TPartnerBoiler::SetRPS(double rps) {
    with_lock (Lock) {
        RPS = rps;
    }
}

void TBoiler::TPartnerBoiler::OnRequestFinished(const NTravelProto::NBoiler::TSearchOffersShortReq& req, TInstant ts, TInstant expireTimestamp, bool error) {
    TKey key;
    key.Currency = req.GetCurrency();
    key.HotelId = THotelId::FromProto(req.GetHotelId());
    key.DateIn = NOrdinalDate::FromString(req.GetCheckInDate());
    key.DateOut = NOrdinalDate::FromString(req.GetCheckOutDate());
    key.Occupancy = TAges::FromOccupancyString(req.GetOccupancy());
    bool greylisted = Owner.Greylist_.IsGreylisted(req.GetPermalink(), key.HotelId);
    with_lock (Lock) {
        TKeyInfo* keyInfo = FindKeyInfoUnlocked(key);
        if (keyInfo) {
            if (ts <= keyInfo->Ts) {
                return;
            }
            OnKeyDelUnlocked(key, keyInfo);
        } else {
            keyInfo = CreateKeyInfoUnlocked(key);
        }
        keyInfo->UpdatePermalink(req.GetPermalink());
        keyInfo->IsGreylisted = greylisted;
        if (error) {
            keyInfo->State = EKeyState::ExistentError;
        } else {
            keyInfo->State = EKeyState::Existent;
        }
        keyInfo->Ts = ts;
        keyInfo->ExpireTs = expireTimestamp;
        AddToKeyExpirationScheduleUnlocked(keyInfo->ExpireTs, key);
        OnKeyAddUnlocked(key, keyInfo);
    }
}

void TBoiler::TPartnerBoiler::OnCacheKeyUse(const ru::yandex::travel::hotels::TOfferCacheReq& message, const TKey& key, TPermalink permalink, TInstant ts, double usage, bool shifted) {
    with_lock (Lock) {
        TKeyInfo* keyInfo = FindKeyInfoUnlocked(key);
        if (keyInfo) {
            OnKeyDelUnlocked(key, keyInfo);
            if (shifted) {
                Counters->NKeysShiftedToExisting.Inc();
            }
        } else {
            keyInfo = CreateKeyInfoUnlocked(key);
            keyInfo->IsGreylisted = Owner.Greylist_.IsGreylisted(permalink, key.HotelId);
            keyInfo->State = EKeyState::Inexistent;
            if (shifted) {
                Counters->NKeysShiftedToNew.Inc();
            }
        }
        keyInfo->UpdatePermalink(permalink);
        keyInfo->Usage.AddUsage(ts, usage, UsageCastTs);
        keyInfo->AllowDateShift = keyInfo->AllowDateShift || message.GetAllowDateShift();
        OnKeyAddUnlocked(key, keyInfo);
    }
}

void TBoiler::TPartnerBoiler::OnInitialBusReadDone() {
    DoKeysRecast(true);
}

TBoiler::TPartnerBoiler::TKeyExpirationItem::TKeyExpirationItem()
    : TotalBytes(GetSizeWithoutKeys())
{
}

void TBoiler::TPartnerBoiler::TKeyExpirationItem::Add(const TKey& key) {
    TotalBytes -= GetSizeWithoutKeys();
    Items.push_back(key);
    TotalBytes += TTotalByteSize<TKey>()(key);
    TotalBytes += GetSizeWithoutKeys();
}

size_t TBoiler::TPartnerBoiler::TKeyExpirationItem::GetSizeWithoutKeys() {
    return sizeof(TKeyExpirationItem) + (Items.capacity() - Items.size()) * sizeof(decltype(Items)::value_type);
}

TKeyInfo* TBoiler::TPartnerBoiler::FindKeyInfoUnlocked(const TKey& key) {
    auto it = AllKeys.find(key);
    if (it != AllKeys.end()) {
        return &it->second;
    }
    return nullptr;
}

TKeyInfo* TBoiler::TPartnerBoiler::CreateKeyInfoUnlocked(const TKey& key) {
    return &AllKeys[key];
}

void TBoiler::TPartnerBoiler::UpdateTotalBytes() {
    Counters->NBytes = Counters->NBytesAllKeys + Counters->NBytesAllKeysForInvalidation + Counters->NBytesKeyExpirationSchedule + Counters->NBytesBoilableKeys;
}

void TBoiler::TPartnerBoiler::MaybeAddToBoilableKeysUnlocked(const TKey& key, const TKeyInfo* keyInfo) {
    if (!keyInfo->IsBoilable()) {
        return;
    }
    TBoilableKeyInfo boilableKeyInfo = TBoilableKeyInfo::FromKeyInfo(key, *keyInfo);
    OnBoilableKeyAdd(boilableKeyInfo);
    bool inserted = BoilableKeys.insert(boilableKeyInfo).second;
    if (!inserted) {
        FATAL_LOG << "Failed to insert key " << key << ", usageCasted " << boilableKeyInfo.UsageCasted << Endl;
        Y_FAIL();
    }
}

void TBoiler::TPartnerBoiler::OnBoilableKeyAdd(const TBoilableKeyInfo& boilableKeyInfo) {
    Counters->NBytesBoilableKeys += TTotalByteSize<TBoilableKeyInfo>()(boilableKeyInfo);
    UpdateTotalBytes();
}

void TBoiler::TPartnerBoiler::OnBoilableKeyDel(const TBoilableKeyInfo& boilableKeyInfo) {
    Counters->NBytesBoilableKeys -= TTotalByteSize<TBoilableKeyInfo>()(boilableKeyInfo);
    UpdateTotalBytes();
}

void TBoiler::TPartnerBoiler::OnKeyAddUnlocked(const TKey& key, TKeyInfo* keyInfo) {
    if (!InitialBusReadDone) {
        return;
    }
    Counters->NBytesAllKeys += TTotalByteSize<TKey>()(key) + TTotalByteSize<TKeyInfo>()(*keyInfo);
    if (keyInfo->State != EKeyState::Inexistent) {
        Counters->NBytesAllKeysForInvalidation += TTotalByteSize<TKey>()(key) + TTotalByteSize<TKeyInfo>()(*keyInfo);
        AllKeysForInvalidation[key] = keyInfo;
    }
    UpdateTotalBytes();
    MaybeAddToBoilableKeysUnlocked(key, keyInfo);
    ModifyCountersUnlocked(keyInfo, 1);
}

void TBoiler::TPartnerBoiler::OnKeyDelUnlocked(const TKey& key, const TKeyInfo* keyInfo, bool removeFromBoilableKeys, bool removeFromInvalidationKeys) {
    if (!InitialBusReadDone) {
        return;
    }
    Counters->NBytesAllKeys -= TTotalByteSize<TKey>()(key) + TTotalByteSize<TKeyInfo>()(*keyInfo);
    if (removeFromInvalidationKeys && keyInfo->State != EKeyState::Inexistent) {
        Counters->NBytesAllKeysForInvalidation -= TTotalByteSize<TKey>()(key) + TTotalByteSize<TKeyInfo>()(*keyInfo);
        AllKeysForInvalidation.erase(key);
    }
    if (removeFromBoilableKeys && keyInfo->IsBoilable()) {
        TBoilableKeyInfo boilableKeyInfo = TBoilableKeyInfo::FromKeyInfo(key, *keyInfo);
        OnBoilableKeyDel(boilableKeyInfo);
        size_t cnt = BoilableKeys.erase(boilableKeyInfo);
        if (cnt != 1) {
            FATAL_LOG << "Failed to delete key " << key << ", usageCasted " << boilableKeyInfo.UsageCasted << ", cnt " << cnt << Endl;
            Y_FAIL();
        }
    }
    ModifyCountersUnlocked(keyInfo, -1);
}

void TBoiler::TPartnerBoiler::ModifyCountersUnlocked(const TKeyInfo* keyInfo, int diff) {
    if (!keyInfo->IsGreylisted) {
        Counters->TotalUsages.AddUsage(keyInfo->Usage, diff, UsageCastTs);
        if (keyInfo->State == EKeyState::Existent) {
            Counters->ExistentUsages.AddUsage(keyInfo->Usage, diff, UsageCastTs);
        }
    }
    switch (keyInfo->State) {
        case EKeyState::Inexistent:
            if (keyInfo->IsGreylisted) {
                Counters->NKeysInexistentGreylisted += diff;
            } else {
                Counters->NKeysInexistent += diff;
            }
            break;
        case EKeyState::InexistentBoiled:
            Counters->NKeysInexistentBoiled += diff;
            break;
        case EKeyState::Existent:
            if (keyInfo->IsGreylisted) {
                Counters->NKeysExistentGreylisted += diff;
            } else {
                Counters->NKeysExistent += diff;
            }
            break;
        case EKeyState::ExistentError:
            Counters->NKeysExistentError += diff;
            break;
    }
    if (keyInfo->AllowDateShift) {
        Counters->NKeysShiftable += diff;
    } else {
        Counters->NKeysUnshiftable += diff;
    }
}

void TBoiler::TPartnerBoiler::AddToKeyExpirationScheduleUnlocked(TInstant when, const TKey& key) {
    auto& scheduleItem = KeyExpirationSchedule[when.Seconds()];
    Counters->NBytesKeyExpirationSchedule -= scheduleItem.TotalBytes;
    scheduleItem.Add(key);
    Counters->NBytesKeyExpirationSchedule += scheduleItem.TotalBytes;
}

void TBoiler::TPartnerBoiler::DoKeyGC(TInstant now) {
    with_lock (Lock) {
        // Пробегаемся по всем ключам, ищем те, у которых последнее обновление слишком давно, истребляем их
        TInstant tsToGc = now - Owner.KeyAgeToGC_;
        for (auto kIt = AllKeys.begin(); kIt != AllKeys.end();) {
            if ((kIt->second.State != EKeyState::Existent) && (kIt->second.Usage.GetUpdateTs() <= tsToGc)) {
                OnKeyDelUnlocked(kIt->first, &kIt->second);
                AllKeys.erase(kIt++);
            } else {
                ++kIt;
            }
        }
    }
}

void TBoiler::TPartnerBoiler::DoBoil(TInstant now, NOrdinalDate::TOrdinalDate minDate) {
    with_lock (Lock) {
        if (AccumulatedRPS  < 1) {// Если RPS не дотрачен в прошлый раз, то не надо добавлять
            AccumulatedRPS += RPS;
        }
        while (AccumulatedRPS >= 1 && !BoilableKeys.empty()) {
            const TKey& key = BoilableKeys.begin()->Key;
            TKeyInfo* keyInfo = FindKeyInfoUnlocked(key);
            if (!keyInfo) {
                FATAL_LOG << "Failed to find key " << key << ", usageCasted " << BoilableKeys.begin()->UsageCasted << " during DoBoil" << Endl;
                Y_FAIL();
            }
            OnKeyDelUnlocked(key, keyInfo, false);
            if (key.DateIn < minDate) {
                AllKeys.erase(key);
                Counters->NKeysObsolete.Inc();
            } else {
                keyInfo->State = EKeyState::InexistentBoiled;
                keyInfo->ExpireTs = now + Owner.KeyBoilExpireTimeout_;
                AddToKeyExpirationScheduleUnlocked(keyInfo->ExpireTs, key);
                OnKeyAddUnlocked(key, keyInfo);
                Owner.Sender_.Send(key, keyInfo->Permalink, "boiler", "", "UsageNow=" + ToString(keyInfo->Usage.GetCastedTo(now)));
                AccumulatedRPS -= 1;
                Counters->NBoils.Inc();
            }
            OnBoilableKeyDel(*BoilableKeys.begin());
            BoilableKeys.erase(BoilableKeys.begin());
        }
    }
}

void TBoiler::TPartnerBoiler::DoKeysExpiration(TInstant now) {
    TInstant tsToGc = now - Owner.KeyAgeToGC_;
    with_lock (Lock) {
        while (!KeyExpirationSchedule.empty() && (KeyExpirationSchedule.begin()->first < now.Seconds())) {
            for (const TKey& key: KeyExpirationSchedule.begin()->second.Items) {
                TKeyInfo* keyInfo = FindKeyInfoUnlocked(key);
                if (keyInfo && keyInfo->ExpireTs <= now) {
                    OnKeyDelUnlocked(key, keyInfo);
                    if (keyInfo->Usage.GetUpdateTs() <= tsToGc) {
                        AllKeys.erase(key);
                    } else {
                        keyInfo->ExpireTs = TInstant::Zero();
                        keyInfo->State = EKeyState::Inexistent;
                        OnKeyAddUnlocked(key, keyInfo);
                    }
                }
            }
            Counters->NBytesKeyExpirationSchedule -= KeyExpirationSchedule.begin()->second.TotalBytes;
            UpdateTotalBytes();
            KeyExpirationSchedule.erase(KeyExpirationSchedule.begin());
        }

        if (Owner.Config_.GetCacheInvalidationEnabled()) {
            TProfileTimer invalidationTimer;
            TVector<TKey> keysToRemove;
            for (auto& [key, keyInfo]: AllKeysForInvalidation) {
                Y_ENSURE(keyInfo->State != EKeyState::Inexistent);
                if (Owner.CacheInvalidationService_.IsInvalidated(TPreKey{key.HotelId, key.Currency},
                                                                  key.DateIn,
                                                                  key.DateOut,
                                                                  keyInfo->Ts)) {
                    keysToRemove.push_back(key);
                    OnKeyDelUnlocked(key, keyInfo, true, false);
                    keyInfo->ExpireTs = TInstant::Zero();
                    keyInfo->State = EKeyState::Inexistent;
                    OnKeyAddUnlocked(key, keyInfo);
                }
            }
            for (auto& key: keysToRemove) {
                AllKeysForInvalidation.erase(key);
            }
            DEBUG_LOG << "Boiler cache invalidation done in " << invalidationTimer.Get() << Endl;
        }
    }
}

void TBoiler::TPartnerBoiler::DoKeysShift(NOrdinalDate::TOrdinalDate dateToShift, size_t& limit) {
    with_lock (Lock) {
        TVector<TKey> keysToShift;
        for (auto it = AllKeys.begin(); it != AllKeys.end() && limit > 0; ++it) {
            if (it->second.AllowDateShift && it->first.DateIn == dateToShift) {
                keysToShift.push_back(it->first);
                --limit;
            }
        }
        for (const TKey& key: keysToShift) {
            TKeyInfo* info = FindKeyInfoUnlocked(key);
            Y_VERIFY(info);
            OnKeyDelUnlocked(key, info);
            TKey shiftedKey = key;
            ShiftKeyDates(shiftedKey);
            TKeyInfo* shiftedInfo = FindKeyInfoUnlocked(shiftedKey);
            if (shiftedInfo) {
                OnKeyDelUnlocked(shiftedKey, shiftedInfo);
                Counters->NKeysShiftedToExisting.Inc();
            } else {
                shiftedInfo = CreateKeyInfoUnlocked(shiftedKey);
                shiftedInfo->State = EKeyState::Inexistent;
                shiftedInfo->IsGreylisted = info->IsGreylisted;
                shiftedInfo->Permalink = info->Permalink;
                Counters->NKeysShiftedToNew.Inc();
            }
            shiftedInfo->AllowDateShift = true;
            info->AllowDateShift = false;

            shiftedInfo->Usage.AddUsage(info->Usage, 1, UsageCastTs);
            info->Usage.Clear();
            OnKeyAddUnlocked(key, info);
            OnKeyAddUnlocked(shiftedKey, shiftedInfo);
        }
        INFO_LOG << "Keys shifted by " << PartnerId << keysToShift.size() << Endl;
    }
}

void TBoiler::TPartnerBoiler::DoKeysRecast(bool first) {
    with_lock (Lock) {
        if (first) {
            InitialBusReadDone.Set();
        }
        UsageCastTs = Owner.GetUsageCastTs();
        Counters->NBytesBoilableKeys = 0;
        UpdateTotalBytes();
        BoilableKeys.clear();
        for (auto kIt = AllKeys.begin(); kIt != AllKeys.end(); ++kIt) {
            TKeyInfo& keyInfo = kIt->second;
            if (!first) {
                // А в первый раз удалять нечего
                OnKeyDelUnlocked(kIt->first, &keyInfo, false);
            }
            keyInfo.Usage.Recast(UsageCastTs);
            OnKeyAddUnlocked(kIt->first, &keyInfo);
        }
    }
}

}// namespace NBoiler
}// namespace NTravel

