#pragma once

#include "data.h"
#include "boiler_data.h"
#include "greylist.h"

#include <travel/hotels/boiler/proto/bus_records.pb.h>
#include <travel/hotels/boiler/proto/config.pb.h>
#include <travel/hotels/proto/data_config/partner.pb.h>
#include <travel/hotels/proto2/bus_messages.pb.h>
#include <travel/hotels/proto2/hotels.pb.h>

#include <travel/hotels/lib/cpp/offer_cache_invalidation/cache_invalidation_service.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/generic/deque.h>
#include <util/thread/factory.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/system/mutex.h>

namespace NTravel {

class TSender;

namespace NBoiler {

class TBoiler : public IThreadFactory::IThreadAble {
public:
    TBoiler(const NTravelProto::NBoiler::TConfig::TBoiler& cfg, TSender& sender, TGreylist& greylist, const TCacheInvalidationService& cacheInvalidationService);
    ~TBoiler();

    void RegisterCounters(NMonitor::TCounterSource& counters);

    void Start();
    void Stop();
    bool IsReady() const;

    void OnInitialBusReadDone();
    void OnUpdatePartnersConfig(const THashMap<EPartnerId, NTravelProto::NConfig::TPartner>& partnersConfig);

    void OnRequestFinished(const NTravelProto::NBoiler::TSearchOffersShortReq& req, TInstant ts, TInstant expireTimestamp, bool error);
    void OnCacheKeyUse(const ru::yandex::travel::hotels::TOfferCacheReq& message, TInstant ts);
private:
    struct TCounters : public NMonitor::TCounterSource, public TThrRefBase {
        NMonitor::TCounter NKeysInexistent;
        NMonitor::TCounter NKeysInexistentGreylisted;
        NMonitor::TCounter NKeysInexistentBoiled;
        NMonitor::TCounter NKeysExistent;
        NMonitor::TCounter NKeysExistentGreylisted;
        NMonitor::TCounter NKeysExistentError;

        NMonitor::TCounter NKeysShiftable;
        NMonitor::TCounter NKeysUnshiftable;

        TKeyUsage ExistentUsages;
        TKeyUsage TotalUsages;
        mutable NMonitor::TDoubleValue BoiledMass;

        NMonitor::TDerivCounter NKeysShiftedToNew;
        NMonitor::TDerivCounter NKeysShiftedToExisting;

        NMonitor::TDerivCounter NBoils;
        NMonitor::TDerivCounter NKeysObsolete;

        NMonitor::TCounter NBytesAllKeys;
        NMonitor::TCounter NBytesAllKeysForInvalidation;
        NMonitor::TCounter NBytesKeyExpirationSchedule;
        NMonitor::TCounter NBytesBoilableKeys;
        NMonitor::TCounter NBytes;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    class TPartnerBoiler : public TThrRefBase {
    public:
        TPartnerBoiler(EPartnerId partnerId, TBoiler& owner);

        void SetRPS(double rps);

        void OnRequestFinished(const NTravelProto::NBoiler::TSearchOffersShortReq& req, TInstant ts, TInstant expireTimestamp, bool error);
        void OnCacheKeyUse(const ru::yandex::travel::hotels::TOfferCacheReq& message, const TKey& key, TPermalink permalink, TInstant ts, double usage, bool shifted);

        void OnInitialBusReadDone();

        void DoKeyGC(TInstant now);
        void DoBoil(TInstant now, NOrdinalDate::TOrdinalDate minDate);
        void DoKeysExpiration(TInstant now);

        void DoKeysShift(NOrdinalDate::TOrdinalDate dateToShift, size_t& limit);
        void DoKeysRecast(bool first);
    private:
        class TKeyExpirationItem {
        public:
            TVector<TKey> Items;
            size_t TotalBytes;

            TKeyExpirationItem();
            void Add(const TKey& key);
        private:
            size_t GetSizeWithoutKeys();
        };

        const EPartnerId PartnerId;
        TBoiler& Owner;
        TAtomicSharedPtr<TCounters> Counters;
        TAtomicFlag InitialBusReadDone;

        TMutex Lock;
        double RPS = 0;
        double AccumulatedRPS = 0;
        TSet<TBoilableKeyInfo> BoilableKeys;
        TInstant UsageCastTs;

        THashMap<TKey, TKeyInfo> AllKeys;
        THashMap<TKey, TKeyInfo*> AllKeysForInvalidation;
        TMap<ui64, TKeyExpirationItem> KeyExpirationSchedule;

        TKeyInfo* FindKeyInfoUnlocked(const TKey& key);
        TKeyInfo* CreateKeyInfoUnlocked(const TKey& key);

        void UpdateTotalBytes();
        void MaybeAddToBoilableKeysUnlocked(const TKey& key, const TKeyInfo* keyInfo);
        void OnBoilableKeyAdd(const TBoilableKeyInfo& boilableKeyInfo);
        void OnBoilableKeyDel(const TBoilableKeyInfo& boilableKeyInfo);
        void OnKeyAddUnlocked(const TKey& key, TKeyInfo* keyInfo);
        void OnKeyDelUnlocked(const TKey& key, const TKeyInfo* keyInfo, bool removeFromBoilableKeys = true, bool removeFromInvalidationKeys = true);
        void ModifyCountersUnlocked(const TKeyInfo* keyInfo, int diff);
        void AddToKeyExpirationScheduleUnlocked(TInstant when, const TKey& key);
    };

    using TPartnerBoilerRef = TIntrusivePtr<TPartnerBoiler>;

    const NTravelProto::NBoiler::TConfig::TBoiler Config_;
    const TDuration KeyBoilExpireTimeout_;
    const TDuration KeyAgeToGC_;
    TGreylist& Greylist_;

    NMonitor::TCounterHypercube<TCounters> Counters_;
    TSender& Sender_;

    TAutoPtr<IThreadFactory::IThread> Thread_;
    TAtomicFlag InitialBusReadDone_;
    TAtomicFlag PartnersConfigGot_;
    TAtomicFlag StopFlag_;
    TAutoEvent WakeUp_;

    ui64 TillKeyGC_;
    ui64 TillKeysExpiration_;

    TConcurrentEntityStorage<EPartnerId, TPartnerBoilerRef> PartnerBoilers_;

    TRWMutex Lock_;
    NOrdinalDate::TOrdinalDate DateToShift_;
    NOrdinalDate::TOrdinalDate MinDate_;
    TInstant UsageCastTs_;

    bool NeedShiftKeys_;

    const TCacheInvalidationService& CacheInvalidationService_;

    TPartnerBoilerRef GetOrCreatePartnerBoiler(EPartnerId pId);

    void DoExecute() override;

    TInstant GetUsageCastTs() const;

    void UpdateDateToShift(TInstant now);
    void DoKeysShift();
    void UpdateMinDate(TInstant now);
};

}// namespace NBoiler
}// namespace NTravel
