#include "boiler_data.h"

#include <travel/hotels/lib/cpp/util/sizes.h>

#include <util/digest/multi.h>
#include <math.h>

namespace NTravel {
namespace NBoiler {

TKeyUsage::TKeyUsage()
    : UpdateTs(TInstant::Zero())
    , Usages(0)
    , UsagesCasted(0)
{
}

void TKeyUsage::AddUsage(TInstant ts, double value, TInstant castTs) {
    i64 secondsDiff = (i64)ts.Seconds() - (i64)UpdateTs.Seconds();
    if (secondsDiff < 0) {
        // ts старее чем у нас в UpdateTs, не меняем UpdateTs, добавляем затухший Usages
        Usages += pow(g_Alpha, -secondsDiff) * value;
    } else {
        if (secondsDiff > 0) {
            Usages *= pow(g_Alpha, secondsDiff);
        }
        Usages += value;
        UpdateTs = ts;
    }
    Recast(castTs);
}

void TKeyUsage::AddUsage(const TKeyUsage& rhs, int coef, TInstant castTs) {
    AddUsage(rhs.UpdateTs, coef * rhs.Usages, castTs);
}

void TKeyUsage::Recast(TInstant castTs) {
    UsagesCasted = GetCastedTo(castTs);
}

void TKeyUsage::Clear() {
    UpdateTs = TInstant::Zero();
    Usages = 0;
    UsagesCasted = 0;
}

double TKeyUsage::GetCastedTo(TInstant ts) const {
    if (ts == TInstant::Zero()) {
        return 0.0;
    }
    return Usages * pow(g_Alpha, (i64)ts.Seconds() - (i64)UpdateTs.Seconds());
}

double TKeyUsage::GetCasted() const {
    return UsagesCasted;
}

TInstant TKeyUsage::GetUpdateTs() const {
    return UpdateTs;
}

//------------------------

void TKeyInfo::UpdatePermalink(TPermalink permalink) {
    if (permalink) {
        Permalink = permalink;
    }
}

bool TKeyInfo::IsBoilable() const {
    return State == EKeyState::Inexistent && !IsGreylisted;
}

size_t TKeyInfo::CalcTotalByteSize() const {
    return sizeof(TKeyInfo);
}
//------------------------

bool TBoilableKeyInfo::operator < (const TBoilableKeyInfo& rhs) const {
    if (UsageCasted != rhs.UsageCasted) {
        // Bigger usages first!
        return UsageCasted > rhs.UsageCasted;
    }
    return Key < rhs.Key;
}

TBoilableKeyInfo TBoilableKeyInfo::FromKeyInfo(const TKey& key, const TKeyInfo& info) {
    return TBoilableKeyInfo{(i64)info.Usage.GetCasted(), key};
}

size_t TBoilableKeyInfo::CalcTotalByteSize() const {
    return sizeof(TBoilableKeyInfo) + TTotalByteSize<TKey>()(Key) - sizeof(Key);
}

//------------------------
void TBoilingThreshold::Update(double newValue) {
    if (IsInited) {
        Value = Value * Lambda + newValue * (1 - Lambda);
    } else {
        IsInited = true;
        Value = newValue;
    }
}

double TBoilingThreshold::GetValue() {
    if (IsInited) {
        return Value;
    } else {
        return NegativeInfinity;
    }
}

const double TBoilingThreshold::Lambda = 0.99;
const double TBoilingThreshold::NegativeInfinity = -1.0;

}// namespace NBoiler
}// namespace NTravel
