#pragma once

#include <travel/hotels/lib/cpp/data/data.h>

#include <library/cpp/containers/concurrent_hash/concurrent_hash.h>

#include <util/generic/ptr.h>
#include <util/system/rwlock.h>

namespace NTravel {

struct TKey {
    NTravelProto::ECurrency Currency;
    THotelId HotelId;
    NOrdinalDate::TOrdinalDate DateIn;
    NOrdinalDate::TOrdinalDate DateOut;
    TAges Occupancy;

    bool operator == (const TKey& rhs) const;
    bool operator <  (const TKey& rhs) const;
    size_t Hash() const;
    size_t CalcTotalByteSize() const;
};

}// namespace NTravel

template <typename K, typename V, size_t BucketCount = 32>
class TConcurrentEntityStorage
{
public:
    V GetOrCreate(K k, std::function<V()> creator) {
        auto& bucket = Map.GetBucketForKey(k);
        {
            TReadGuard g(bucket.GetMutex());
            auto it = bucket.GetMap().find(k);
            if (it != bucket.GetMap().end()) {
                return it->second;
            }
        }
        {
            TWriteGuard g(bucket.GetMutex());
            auto it = bucket.GetMap().find(k);
            if (it == bucket.GetMap().end()) {
                it = bucket.GetMap().emplace(k, creator()).first;
            }
            return it->second;
        }
    }

    template<typename TAction>
    void ForEach(TAction&& action) {
        for (auto& bucket: Map.Buckets) {
            TReadGuard g(bucket.GetMutex());
            for (auto it = bucket.GetMap().begin(); it != bucket.GetMap().end(); ++it) {
                action(it->second);
            }
        }
    }
private:
    TConcurrentHashMap<K, V, BucketCount, TRWMutex> Map;
};

//----------------------------------

template <>
struct THash<NTravel::TKey> {
    inline size_t operator() (const NTravel::TKey& v) const { return v.Hash(); }
};
