#include "grey_warmer.h"
#include "sender.h"

#include <library/cpp/logger/global/global.h>

#include <util/generic/algorithm.h>

#include <algorithm>

namespace NTravel {
namespace NGreyWarmer {

void TGreyWarmer::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    Y_UNUSED(ct);
}

void TGreyWarmer::TPartnerCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NDuplicatedOriginalIdsFromStateBus));

    ct->insert(MAKE_COUNTER_PAIR(CurrnetHotelIndex));

    ct->insert(MAKE_COUNTER_PAIR(NPartnerHotels));
    ct->insert(MAKE_COUNTER_PAIR(NPartnerHotelsWarmed));

    ct->insert(MAKE_COUNTER_PAIR(NPartnerHotelsHasOffers));
    ct->insert(MAKE_COUNTER_PAIR(NPartnerHotelsHasNoOffers));
    ct->insert(MAKE_COUNTER_PAIR(NPartnerHotelsStatusUnknown));

    ct->insert(MAKE_COUNTER_PAIR(NHotelIds));
    ct->insert(MAKE_COUNTER_PAIR(NHotelIdsHasOffers));

    ct->insert(MAKE_COUNTER_PAIR(EffectiveRps));
}

TString TGreyWarmer::GetOfferCacheClientId() {
    return "grey-warmer";
}

TGreyWarmer::TGreyWarmer(const TConfig& config, TYtQueueWriter& stateBusWriter, TSender& sender)
    : Config_(config)
    , StateBusWriter_(stateBusWriter)
    , Sender_(sender)
    , PartnerCounters_({"partner"})
    , TableCache_("GreyWarmerHotelsGreylist", config.GetHotelsGreylist())
{
    auto conv = [](const NYT::TNode& node, TProto* proto) {
        auto& hotelId = *proto->MutableHotelId();
        hotelId.SetPartnerId(ExtractPartnerIdFromYtRow(node));
        hotelId.SetOriginalId(ExtractOriginalIdFromYtRow(node));
        proto->SetPermalink(ExtractPermalinkFromYtRow(node));
    };
    auto data = [this](const TProto& proto) {
        auto hotelId = THotelId::FromProto(proto.GetHotelId());
        auto permalink = TPermalink(proto.GetPermalink());
        GetOrCreatePartnerWarmer(hotelId.PartnerId)->AddKey(hotelId.OriginalId, permalink);
    };
    auto finish = [this](bool ok, bool /*initial*/) {
        GenerateKeys(ok);
    };
    TableCache_.SetCallbacks(conv, data, finish);
}

void TGreyWarmer::RegisterCounters(NMonitor::TCounterSource& source) {
    source.RegisterSource(&Counters_, "GreyWarmerCommon");
    source.RegisterSource(&PartnerCounters_, "GreyWarmerPerPartner");
    TableCache_.RegisterCounters(source, "GreyWarmerBlacklistTableCache");
}

void TGreyWarmer::Start() {
    SchedulerThread_ = SystemThreadFactory()->Run([this]{ SchedulerThreadLoop(); });

    TableCache_.Start();
}

void TGreyWarmer::Stop() {
    TableCache_.Stop();

    StopFlag_.Set();
    if (SchedulerThread_) {
        SchedulerWakeUp_.Signal();
        SchedulerThread_->Join();
        SchedulerThread_.Destroy();
    }
}

bool TGreyWarmer::IsReady() const {
    return InitialBusReadDone_ && PartnersConfigGot_ && TableCache_.IsReady();
}

void TGreyWarmer::OnInitialBusReadDone() {
    PartnerWarmers_.ForEach([](TPartnerWarmerRef partnerWarmer) { partnerWarmer->OnInitialBusReadDone(); });
    InitialBusReadDone_.Set();
}

void TGreyWarmer::OnUpdatePartnersConfig(const THashMap<EPartnerId, NTravelProto::NConfig::TPartner>& partnersConfig) {
    PartnerWarmers_.ForEach([](TPartnerWarmerRef partnerWarmer){
        partnerWarmer->SetRPS(0);
    });
    for (auto pcIt = partnersConfig.begin(); pcIt != partnersConfig.end(); ++pcIt) {
        GetOrCreatePartnerWarmer(pcIt->first)->SetRPS(pcIt->second.GetGreyWarmerRPS());
    }
    PartnersConfigGot_.Set();
}

void TGreyWarmer::OnBeginWarmMessage(const ru::yandex::travel::hotels::TGreyWarmerKeyBeginWarmMessage& message, TInstant ts) {
    OnNewHotelId(message.GetHotelId(), false, ts);
}

void TGreyWarmer::OnOffersFoundMessage(const ru::yandex::travel::hotels::TGreyWarmerKeyOffersFoundMessage& message, TInstant ts) {
    OnNewHotelId(message.GetHotelId(), true, ts);
}

void TGreyWarmer::OnRequestFinished(const NTravelProto::NBoiler::TSearchOffersShortReq& req, bool hasOffers, bool error) {
    if (req.GetOfferCacheClientId() != GetOfferCacheClientId()) {
        return;
    }
    auto hotelId = THotelId::FromProto(req.GetHotelId());
    GetOrCreatePartnerWarmer(hotelId.PartnerId)->OnRequestFinished(hotelId.OriginalId, hasOffers, error);
}

TGreyWarmer::TPartnerWarmerRef TGreyWarmer::GetOrCreatePartnerWarmer(EPartnerId pId) {
    return PartnerWarmers_.GetOrCreate(pId, [this, pId](){
       return new TPartnerWarmer(pId, *this);
    });
}

void TGreyWarmer::GenerateKeys(bool ok) {
    PartnerWarmers_.ForEach([ok](TPartnerWarmerRef partnerWarmer) { partnerWarmer->GenerateKeys(ok); });
}

void TGreyWarmer::SchedulerThreadLoop() {
    auto nextAction = TInstant::Now();
    while (!StopFlag_) {
        nextAction += TDuration::Seconds(1);
        if (SchedulerWakeUp_.WaitD(nextAction)) {
            continue;
        }
        if (IsReady()) {
            const NOrdinalDate::TOrdinalDate today = NOrdinalDate::FromInstant(nextAction + TDuration::Hours(Config_.GetTodayHoursOffset()));
            PartnerWarmers_.ForEach([today](TPartnerWarmerRef partnerWarmer) {
                partnerWarmer->DoWarm(today);
            });
        }
        PartnerWarmers_.ForEach([nextAction](TPartnerWarmerRef partnerWarmer) {
            partnerWarmer->DoExpiration(nextAction);
        });
        PartnerWarmers_.ForEach([](TPartnerWarmerRef partnerWarmer) {
            partnerWarmer->UpdateCounters();
        });
    }
}

void TGreyWarmer::OnNewHotelId(const NTravelProto::THotelId& hotelIdProto, bool hasOffers, TInstant ts) {
    auto hotelId = THotelId::FromProto(hotelIdProto);
    GetOrCreatePartnerWarmer(hotelId.PartnerId)->OnNewOriginalId(hotelId.OriginalId, hasOffers, ts);
}

TGreyWarmer::TPartnerWarmer::TPartnerWarmer(EPartnerId partnerId, TGreyWarmer& owner)
    : PartnerId_(partnerId)
    , Counters_(owner.PartnerCounters_.GetOrCreate({ToString(partnerId)}))
    , Owner_(owner)
    , IsFullyWarmed_(false)
    , RPS_(0)
    , AccumulatedRps_(0)
{
    SetCurrentHotelIndex(0);
}

void TGreyWarmer::TPartnerWarmer::SetRPS(double rps) {
    with_lock (Lock_) {
        RPS_ = rps;
    }
}

void TGreyWarmer::TPartnerWarmer::AddKey(const TOriginalId& originalId, TPermalink permalink) {
    TempHotels_.emplace_back(originalId, permalink);
}

void TGreyWarmer::TPartnerWarmer::GenerateKeys(bool ok) {
    if (!ok) {
        TempHotels_.clear();
        return;
    }
    if (TempHotels_.empty()) {
        WARNING_LOG << "New blacklist table for partner " << PartnerId_ <<  " is empty" << Endl;
    } else {
        SortUniqueBy(TempHotels_, [](auto&& hotel) { return hotel.first; });
    }
    with_lock (Lock_) {
        IsFullyWarmed_ = false;
        Hotels_.swap(TempHotels_);
        CalculateCurrentHotelIndex();
    }
    TempHotels_.clear();
}

void TGreyWarmer::TPartnerWarmer::DoWarm(const NOrdinalDate::TOrdinalDate today) {
    with_lock (Lock_) {
        if (Hotels_.empty()) {
            return;
        }
        if (IsFullyWarmed_) {
            return;
        }
        if (AccumulatedRps_ < 1) {
            AccumulatedRps_ += RPS_;
            return;
        }
        THotelId hotelId;
        hotelId.PartnerId = PartnerId_;
        bool isRpsSpent = false;
        size_t currentHotelIndex = CurrentHotelIndex_;
        size_t numberOfHotelsToCheck = Hotels_.size();
        do {
            const auto& [originalId, permalink] = Hotels_[currentHotelIndex];
            currentHotelIndex = (currentHotelIndex + 1) % Hotels_.size();

            if (WarmedHotels_.contains(originalId)) {
                continue;
            }

            if (JustWarmedHotelsGen1_.contains(originalId) || JustWarmedHotelsGen2_.contains(originalId)) {
                continue;
            }
            JustWarmedHotelsGen1_.insert(originalId);

            hotelId.OriginalId = originalId;

            TKey key;
            key.Currency = DefaultCurrency_;
            key.HotelId = hotelId;
            key.DateIn = today + 1;
            key.DateOut = today + 2;
            key.Occupancy = DefaultOccupancy_;
            Owner_.Sender_.Send(key, permalink, GetOfferCacheClientId(), "", NOrdinalDate::ToString(today));
            AccumulatedRps_ -= 1.0f;
            isRpsSpent = true;
            Counters_->EffectiveRps.Inc();

            ru::yandex::travel::hotels::TGreyWarmerKeyBeginWarmMessage beginWarmMessage;
            hotelId.ToProto(beginWarmMessage.MutableHotelId());
            Owner_.StateBusWriter_.Write(beginWarmMessage);
        } while ((--numberOfHotelsToCheck != 0) && !(AccumulatedRps_ < 1));
        if (numberOfHotelsToCheck == 0) {
            IsFullyWarmed_ = true;
        }
        SetCurrentHotelIndex(currentHotelIndex);
        if (isRpsSpent) {
            AccumulatedRps_ += RPS_;
        }
    }
}

void TGreyWarmer::TPartnerWarmer::OnInitialBusReadDone() {
    with_lock (Lock_) {
        CalculateCurrentHotelIndex();
    }
}

void TGreyWarmer::TPartnerWarmer::OnNewOriginalId(const TOriginalId& originalId, bool hasOffers, TInstant ts) {
    with_lock (Lock_) {
        const auto hotel = LowerBoundBy(Hotels_.cbegin(), Hotels_.cend(), originalId, [](auto&& hotel) { return hotel.first; });
        if ((hotel != Hotels_.cend()) && !(hotel->first < originalId)) {
            Y_VERIFY((std::next(hotel) == Hotels_.cend()) || (originalId < std::next(hotel)->first));
            // advance CurrentHotelIndex_ if another instance wrote to bus an OriginalId which can be warmed by the current instance in the near future
            const auto hotelIndex = size_t(std::distance(Hotels_.cbegin(), hotel));
            if (((hotelIndex + Hotels_.size()) - CurrentHotelIndex_) % Hotels_.size() < Hotels_.size() / 2) {
                SetCurrentHotelIndex(hotelIndex);
            }
        }

        TWarmedHotel newWarmedHotel{ts, ts + TDuration::Seconds(Owner_.Config_.GetMaxKeyAgeSec()), hasOffers};
        auto [wh, isInserted] = WarmedHotels_.emplace(originalId, newWarmedHotel);
        if (isInserted) {
            Counters_->NHotelIds.Inc();
            if (hasOffers) {
                Counters_->NHotelIdsHasOffers.Inc();
            }

            WarmedHotelsExpirationSchedule_.emplace(newWarmedHotel.ExpiredAt, originalId);
        } else {
            auto& warmedHotel = wh->second;
            Counters_->NDuplicatedOriginalIdsFromStateBus.Inc();
            auto range = WarmedHotelsExpirationSchedule_.equal_range(warmedHotel.ExpiredAt);
            Y_VERIFY(range.first != range.second);
            auto duplicate = FindIf(range.first, range.second, [&originalId](const auto& item) -> bool { return item.second == originalId; });
            Y_VERIFY(duplicate != range.second);
            Y_VERIFY(FindIf(std::next(duplicate), range.second, [&originalId](const auto& item) -> bool { return item.second == originalId; }) == range.second);
            if (duplicate->first < newWarmedHotel.ExpiredAt) {
                WarmedHotelsExpirationSchedule_.erase(duplicate);
                WarmedHotelsExpirationSchedule_.emplace(newWarmedHotel.ExpiredAt, originalId);
                warmedHotel.ExpiredAt = newWarmedHotel.ExpiredAt;
            }
            if (warmedHotel.CreatedAt < newWarmedHotel.CreatedAt) {
                warmedHotel.CreatedAt = newWarmedHotel.CreatedAt;
            }
            if (hasOffers && !warmedHotel.HasOffers) {
                warmedHotel.HasOffers = true;
                Counters_->NHotelIdsHasOffers.Inc();
            }
        }
    }
}

void TGreyWarmer::TPartnerWarmer::DoExpiration(TInstant now) {
    with_lock (Lock_) {
        const auto beg = WarmedHotelsExpirationSchedule_.cbegin();
        const auto end = WarmedHotelsExpirationSchedule_.upper_bound(now);
        for (auto it = beg; it != end; ++it) {
            const auto& originalId = it->second;
            auto [warmedHotel, nextWarmedHotel] = WarmedHotels_.equal_range(originalId);
            Y_VERIFY(std::distance(warmedHotel, nextWarmedHotel) == 1);
            if (warmedHotel->second.HasOffers) {
                Counters_->NHotelIdsHasOffers.Dec();
            }
            WarmedHotels_.erase(warmedHotel);
            Counters_->NHotelIds.Dec();
        }
        WarmedHotelsExpirationSchedule_.erase(beg, end);
        if (beg != end) {
            IsFullyWarmed_ = false;
        }
        for (const auto& justWarmedHotel: JustWarmedHotelsGen2_) {
            if (!WarmedHotels_.contains(justWarmedHotel)) {
                WARNING_LOG << "OriginalId " << justWarmedHotel << " for PartnerId " << PartnerId_ << " is not warmed within two ticks from start" << Endl;
            }
        }
        JustWarmedHotelsGen2_.clear();
        JustWarmedHotelsGen1_.swap(JustWarmedHotelsGen2_);
    }
}

void TGreyWarmer::TPartnerWarmer::OnRequestFinished(const TOriginalId& originalId, bool hasOffers, bool error) const {
    with_lock (Lock_) {
        auto hotel = LowerBoundBy(Hotels_.cbegin(), Hotels_.cend(), originalId, [](auto&& hotel) { return hotel.first; });
        if ((hotel != Hotels_.cend()) && !(hotel->first < originalId)) {
            if (hasOffers) {
                OnOffersFound(originalId);
                Counters_->NPartnerHotelsHasOffers.Inc();
            } else if (error) {
                Counters_->NPartnerHotelsStatusUnknown.Inc();
            } else {
                Counters_->NPartnerHotelsHasNoOffers.Inc();
            }
        }
    }
}

void TGreyWarmer::TPartnerWarmer::UpdateCounters() const {
    with_lock (Lock_) {
        size_t partnerHotelsWarmed = 0;
        for (const auto& [originalId, permalink]: Hotels_) {
            if (WarmedHotels_.contains(originalId)) {
                ++partnerHotelsWarmed;
            }
        }
        Counters_->NPartnerHotels = Hotels_.size();
        Counters_->NPartnerHotelsWarmed = partnerHotelsWarmed;
    }
}

void TGreyWarmer::TPartnerWarmer::SetCurrentHotelIndex(size_t currentHotelIndex) {
    CurrentHotelIndex_ = currentHotelIndex;
    Counters_->CurrnetHotelIndex = CurrentHotelIndex_;
}

void TGreyWarmer::TPartnerWarmer::CalculateCurrentHotelIndex() {
    const auto end = WarmedHotels_.end();
    auto warmedHotel = end;
    size_t partnerHotelsWarmed = 0;
    size_t latestWarmedHotelIndex = 0;
    for (size_t hotelIndex = 0; hotelIndex < Hotels_.size(); ++hotelIndex) {
        const auto& originalId = Hotels_[hotelIndex].first;
        if (auto it = WarmedHotels_.find(originalId); it != end) {
            ++partnerHotelsWarmed;
            if ((warmedHotel == end) || (warmedHotel->second.CreatedAt < it->second.CreatedAt)) {
                latestWarmedHotelIndex = hotelIndex;
                warmedHotel = it;
            }
        }
    }
    Counters_->NPartnerHotels = Hotels_.size();
    Counters_->NPartnerHotelsWarmed = partnerHotelsWarmed;
    SetCurrentHotelIndex(latestWarmedHotelIndex);
}

void TGreyWarmer::TPartnerWarmer::OnOffersFound(const TOriginalId& originalId) const {
    ru::yandex::travel::hotels::TGreyWarmerKeyOffersFoundMessage offersFoundMessage;
    THotelId hotelId;
    hotelId.PartnerId = PartnerId_;
    hotelId.OriginalId = originalId;
    hotelId.ToProto(offersFoundMessage.MutableHotelId());
    Owner_.StateBusWriter_.Write(offersFoundMessage);
}

}
}
