#pragma once

#include "data.h"

#include <travel/hotels/boiler/proto/bus_records.pb.h>
#include <travel/hotels/boiler/proto/config.pb.h>
#include <travel/hotels/proto/data_config/partner.pb.h>
#include <travel/hotels/proto2/bus_messages.pb.h>
#include <travel/hotels/proto2/hotels.pb.h>

#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/yt/table_cache.h>
#include <travel/hotels/lib/cpp/yt/queue_writer.h>
#include <travel/hotels/lib/cpp/util/flag.h>

#include <util/thread/factory.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <util/generic/ptr.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/system/mutex.h>

namespace NTravel {

class TSender;

namespace NGreyWarmer {

class TGreyWarmer {
public:
    using TConfig = NTravelProto::NBoiler::TConfig::TGreyWarmer;

    TGreyWarmer(const TConfig& config, TYtQueueWriter& stateBusWriter, TSender& sender);

    void RegisterCounters(NMonitor::TCounterSource& counters);

    void Start();
    void Stop();
    bool IsReady() const;

    void OnInitialBusReadDone();
    void OnUpdatePartnersConfig(const THashMap<EPartnerId, NTravelProto::NConfig::TPartner>& partnersConfig);

    void OnBeginWarmMessage(const ru::yandex::travel::hotels::TGreyWarmerKeyBeginWarmMessage& message, TInstant ts);
    void OnOffersFoundMessage(const ru::yandex::travel::hotels::TGreyWarmerKeyOffersFoundMessage& message, TInstant ts);
    void OnRequestFinished(const NTravelProto::NBoiler::TSearchOffersShortReq& req, bool hasOffers, bool error);
private:
    struct TCounters: public NMonitor::TCounterSource {
        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TPartnerCounters: public NMonitor::TCounterSource, public TThrRefBase {
        NMonitor::TDerivCounter NDuplicatedOriginalIdsFromStateBus;

        NMonitor::TCounter CurrnetHotelIndex;

        NMonitor::TCounter NPartnerHotels;
        NMonitor::TCounter NPartnerHotelsWarmed; // can be greater then NPartnerHotels after table switch

        NMonitor::TDerivCounter NPartnerHotelsHasOffers;
        NMonitor::TDerivCounter NPartnerHotelsHasNoOffers;
        NMonitor::TDerivCounter NPartnerHotelsStatusUnknown;

        NMonitor::TCounter NHotelIds;
        NMonitor::TCounter NHotelIdsHasOffers;

        NMonitor::TDerivCounter EffectiveRps;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    using TProto = NTravelProto::THotelIdWithPermalink;
    using THotelIdWithPermalink = std::pair<THotelId, TPermalink>;
    using TOriginalId = TString;

    class TPartnerWarmer : public TThrRefBase {
    public:
        TPartnerWarmer(EPartnerId partnerId, TGreyWarmer& owner);

        void SetRPS(double rps);

        void AddKey(const TOriginalId& originalId, TPermalink permalink);
        void GenerateKeys(bool ok);
        void DoWarm(const NOrdinalDate::TOrdinalDate today);

        void OnInitialBusReadDone();
        void OnNewOriginalId(const TOriginalId& originalId, bool hasOffers, TInstant ts);
        void DoExpiration(TInstant now);

        void OnRequestFinished(const TOriginalId& originalId, bool hasOffers, bool error) const;

        void UpdateCounters() const;
    private:
        using THotel = std::pair<TOriginalId, TPermalink>;

        struct TWarmedHotel {
            TInstant CreatedAt;
            TInstant ExpiredAt;
            bool HasOffers;
        };

        const EPartnerId PartnerId_;
        const NTravelProto::ECurrency DefaultCurrency_ = NTravelProto::C_RUB;
        const TAges DefaultOccupancy_ = TAges::FromOccupancyString("2");

        const TAtomicSharedPtr<TPartnerCounters> Counters_;

        TGreyWarmer& Owner_;

        TVector<THotel> TempHotels_;

        TMutex Lock_;
        bool IsFullyWarmed_;
        double RPS_;
        double AccumulatedRps_;
        TVector<THotel> Hotels_;
        size_t CurrentHotelIndex_; // should be modified by means of SetCurrentHotelIndex only
        THashMap<TOriginalId, TWarmedHotel> WarmedHotels_;
        TMultiMap<TInstant/*expireTs*/, const TOriginalId> WarmedHotelsExpirationSchedule_;
        THashSet<TOriginalId> JustWarmedHotelsGen1_, JustWarmedHotelsGen2_; // do not warm hotels that have recently started to warm. All duplicates occur within 2 seconds (see HOTELS-4349)

        void SetCurrentHotelIndex(size_t currentHotelIndex);
        void CalculateCurrentHotelIndex(); // assign index of last known warmed partner hotel to CurrentHotelIndex_

        void OnOffersFound(const TOriginalId& originalId) const;
    };
    using TPartnerWarmerRef = TIntrusivePtr<TPartnerWarmer>;

    const TConfig Config_;

    TConcurrentEntityStorage<EPartnerId, TPartnerWarmerRef> PartnerWarmers_;

    TYtQueueWriter& StateBusWriter_;
    TSender& Sender_;

    TCounters Counters_;
    NMonitor::TCounterHypercube<TPartnerCounters> PartnerCounters_;

    THolder<IThreadFactory::IThread> SchedulerThread_;
    TAtomicFlag StopFlag_;
    TAutoEvent SchedulerWakeUp_;

    TYtTableCache<TProto> TableCache_;

    TAtomicFlag InitialBusReadDone_;
    TAtomicFlag PartnersConfigGot_;

    static TString GetOfferCacheClientId();

    TPartnerWarmerRef GetOrCreatePartnerWarmer(EPartnerId pId);

    void GenerateKeys(bool ok);

    void SchedulerThreadLoop();

    void OnNewHotelId(const NTravelProto::THotelId& hotelIdProto, bool hasOffers, TInstant ts);
};

}
}
