#include "sender.h"

#include <library/cpp/logger/global/global.h>

#include <util/generic/guid.h>
#include <util/system/hostname.h>
#include <util/digest/multi.h>

#include <math.h>

namespace NTravel {

void TSender::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NotEnabled));
    ct->insert(MAKE_COUNTER_PAIR(NGrpcErrors));
    ct->insert(MAKE_COUNTER_PAIR(NSubRespErrors));
}

TSender::TSender(const NTravelProto::NAppConfig::TConfigGrpcClient& offerCacheCfg, const NTravelProto::NBoiler::TConfig::TSender& senderCfg)
    : SendDelaySec_(senderCfg.GetSendDelaySec())
    , SendMaxDelaySec_(0)
    , SendOffsetSec_(0)
    , Enabled_(false)
    , YpAutoResolver_(senderCfg.GetYpAutoResolver())
    , OfferCacheClient_(offerCacheCfg)
{
    size_t delay = 0;
    for (const auto& src: senderCfg.GetClientList().GetClient()) {
        ClientSendDelaySec_[src] = delay;
        delay += senderCfg.GetClientSendDelaySec();
    }
    YpAutoResolver_.SetCallback([this] (const TVector<TYpConnectionInfo>& endpoints) {
        TVector<TString> hosts(endpoints.size());
        std::transform(endpoints.begin(), endpoints.end(), hosts.begin(), [] (const auto& ep) { return ep.GetFqdn(); });
        UpdateHosts(hosts);
    });
}

TSender::~TSender() {
    Stop();
}

void TSender::RegisterCounters(NMonitor::TCounterSource& counters) {
    counters.RegisterSource(&Counters_, "Sender");
}

void TSender::Start() {
    Thread_ = SystemThreadFactory()->Run(this);
    YpAutoResolver_.Start();
}

void TSender::Stop() {
    YpAutoResolver_.Stop();
    StopFlag_.Set();
    WakeUp_.Signal();
    if (Thread_ != nullptr) {
        Thread_->Join();
        Thread_.Destroy();
    }
    OfferCacheClient_.Shutdown();
}

TString TSender::GetOfferCacheClientId(const TString& client, const TString& clientSuffix) const {
    TString ocClientId = client;
    if (clientSuffix) {
        ocClientId += "-" + clientSuffix;
    }
    return ocClientId;
}

void TSender::Send(const TKey& key, TPermalink permalink, const TString& client, const TString& clientSuffix, const TString& comment) {
    size_t sendMaxDelaySec, sendOffsetSec;
    with_lock (YpLock_) {
        if (!Enabled_) {
            return;
        }
        sendMaxDelaySec = SendMaxDelaySec_;
        sendOffsetSec = SendOffsetSec_;
    }
    TSendItem si;
    si.Key = key;
    si.Permalink = permalink;
    si.Client = client;
    si.ClientSuffix = clientSuffix;
    si.Comment = comment;
    // сложная система тросов и лебёдок.
    // Каждый ключик посылается в момент, зависящий от хэша ключа и от смещения текущего хоста, и от Client-а
    // Причем эти моменты выровнены на сетку времени
    // а именно, CurrentOffset = (hash(key) + sendOffset - now.Seconds()) % sendMaxDelay
    // Смысл всего этого в том, чтобы разные хосты не могли послать один и тот же ключ одновременно
    size_t sourceAdd = 0;
    auto it = ClientSendDelaySec_.find(client);
    if (it == ClientSendDelaySec_.end()) {
        WARNING_LOG << "Unknown Client " << client << Endl;
    } else {
        sourceAdd = it->second;
    }
    size_t sendTimepoint = (sendOffsetSec + key.Hash() + sourceAdd) % sendMaxDelaySec;
    size_t sendOffset = (sendTimepoint + sendMaxDelaySec - Now().Seconds() % sendMaxDelaySec) % sendMaxDelaySec;
    with_lock (QueueLock_) {
        if (SendQueue_.size() < (sendOffset + 1)) {
            SendQueue_.resize(sendOffset + 1);
        }
        SendQueue_[sendOffset].push_back(si);
    }
}

void TSender::DoExecute() {
    while (!StopFlag_) {
        DoSend();
        WakeUp_.WaitT(TDuration::Seconds(1));
    }
}

void TSender::DoSend() {
    TVector<TSendItem> toSend;
    with_lock (QueueLock_) {
        if (!SendQueue_.empty()) {
            toSend.swap(SendQueue_.front());
            SendQueue_.pop_front();
        }
    }
    if (toSend.empty()) {
        return;
    }
    NTravelProto::TSearchOffersRpcReq rpcReq;
    for (const TSendItem& si: toSend) {
        auto sub = rpcReq.AddSubrequest();
        sub->SetId(CreateGuidAsString());
        si.Key.HotelId.ToProto(sub->MutableHotelId());
        sub->SetPermalink(si.Permalink);
        sub->SetCurrency(si.Key.Currency);
        sub->SetCheckInDate(NOrdinalDate::ToString(si.Key.DateIn));
        sub->SetCheckOutDate(NOrdinalDate::ToString(si.Key.DateOut));
        sub->SetOccupancy(si.Key.Occupancy.ToOccupancyString());
        sub->SetRequestClass(NTravelProto::RC_BACKGROUND);
        TString ocClientId = GetOfferCacheClientId(si.Client, si.ClientSuffix);
        sub->MutableAttribution()->SetOfferCacheClientId(ocClientId);
        sub->SetOfferCacheUseCache(true);
        sub->SetOfferCacheUseSearcher(true);
        INFO_LOG << "Sending (" << ocClientId << ") key " << si.Key << ", subReq ID " << sub->GetId() << ", comment: " << si.Comment << Endl;
    }
    INFO_LOG << "Sending request with " << rpcReq.SubrequestSize() << " subreqs: " << GetSearchOffersRpcReqIds(rpcReq) << Endl;

    OfferCacheClient_.Request<NTravelProto::TSearchOffersRpcReq, NTravelProto::TSearchOffersRpcRsp, &NTravelProto::NOfferCacheGrpc::OfferCacheServiceV1::Stub::AsyncSearchOffers>
            (rpcReq, NGrpc::TClientMetadata(), [this, rpcReq](const TString& grpcError, const TString& remoteFQDN, const NTravelProto::TSearchOffersRpcRsp& resp) {
        if (grpcError) {
            ERROR_LOG << "Boil request has gRPC error: " << grpcError << Endl;
            Counters_.NGrpcErrors.Inc();
        } else {
            INFO_LOG << "Boil request finished, remote FQDN " << remoteFQDN << Endl;
            for (size_t pos = 0; pos < resp.SubresponseSize(); ++pos) {
                const auto& subResp = resp.GetSubresponse(pos);
                if (subResp.HasError()) {
                    Counters_.NSubRespErrors.Inc();
                    TString requestId;
                    if (pos < rpcReq.SubrequestSize()) {
                        requestId = rpcReq.GetSubrequest(pos).GetId();
                    }
                    ERROR_LOG << "Boil response has error in subresponse #" << pos << ", RequestId: " << requestId << ": " << subResp.GetError() << Endl;
                }
            }
        }
    });
}

void TSender::UpdateHosts(const TVector<TString>& hosts) {
    with_lock (YpLock_) {
        SendMaxDelaySec_ = Max(1ul, SendDelaySec_ * hosts.size());
        SendOffsetSec_ = 0;
        Enabled_ = false;
        for (const auto& host : hosts) {
            if (host == FQDNHostName()) {
                Enabled_ = true;
                break;
            }
            SendOffsetSec_ += SendDelaySec_;
        }
        if (Enabled_) {
            INFO_LOG << "Send max delay is " << SendMaxDelaySec_ << ", offset is " << SendOffsetSec_ << Endl;
            Counters_.NotEnabled = 0;
        } else {
            ERROR_LOG << "Host was not found nor enabled, FQDN " << FQDNHostName() << Endl;
            Counters_.NotEnabled = 1;
        }
    }
}

}// namespace NTravel

