#include "service.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/proto2/bus_messages.pb.h>
#include <travel/hotels/boiler/proto/bus_records.pb.h>

#include <library/cpp/logger/global/global.h>

#include <util/datetime/base.h>
#include <util/charset/wide.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/deque.h>
#include <util/generic/algorithm.h>
#include <util/generic/guid.h>
#include <util/system/hostname.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/string_utils/quote/quote.h>

namespace NTravel {

TService::TCounters::TCounters(TService& service)
    : Service_(service)
{
}

void TService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    IsReady = Service_.IsReady() ? 1: 0;
    ct->insert(MAKE_COUNTER_PAIR(IsReady));
    ct->insert(MAKE_COUNTER_PAIR(OfferBusProcessBoilerMicros));
    ct->insert(MAKE_COUNTER_PAIR(OfferBusProcessGreyWarmerMicros));
    ct->insert(MAKE_COUNTER_PAIR(OfferReqBusProcessBoilerMicros));
    ct->insert(MAKE_COUNTER_PAIR(WarmerBusProcessMicros));

}

TService::TService(const NTravelProto::NBoiler::TConfig& pbCfg, const TString& environment)
    : Config_(pbCfg)
    , Counters_(*this)
    , Greylist_(Config_.GetHotelsGreylist())
    , CacheInvalidationService_(pbCfg.GetCacheInvalidationService())
    // Более новые записи вытесняют более старые.
    // Если более новая запись оказалась с коротким TTL, её всё равно надо прочитать, чтобы узнать, что старая запись невалидна.
    // https://st.yandex-team.ru/TRAVELBACK-866#5f05a3b10234bc593a755fa5
    , OfferBus_(pbCfg.GetOfferBus(), "OfferBus", TYtQueueReaderOptions().SetCheckExpireTimestamp(false))
    , OfferReqBus_(pbCfg.GetOfferReqBus(), "OfferReqBus")
    , YtConfigPartners_("YtConfigPartners", pbCfg.GetYtConfigPartners())
    , WarmerStateBusWriter_(pbCfg.GetWarmerStateBus().GetWriter(), "WarmerStateBusWriter")
    , WarmerStateBusReader_(pbCfg.GetWarmerStateBus().GetReader(), "WarmerStateBusReader")
    , Sender_(pbCfg.GetOfferCache(), pbCfg.GetSender())
    , Boiler_(pbCfg.GetBoiler(), Sender_, Greylist_, CacheInvalidationService_)
    , GreyWarmer_(pbCfg.GetGreyWarmer(), WarmerStateBusWriter_, Sender_)
    , ReadyBusCount_(0)
    , RestartDetector_(Config_.GetOther().GetRestartDetectorStateFile())
{
    Y_UNUSED(environment);
    Http_.AddHandler("/ping_nanny",      NHttp::ExternalWithoutTvm(), this, &TService::OnPingNanny);
    Http_.AddHandler("/setlog",          NHttp::Local(),    this, &TService::OnSetLogLevel);

    CountersPage_.AddToHttpService(MonWebService_);

    CountersPage_.RegisterSource(&Counters_, "Server");
    Http_.RegisterCounters(CountersPage_);
    OfferBus_.RegisterCounters(CountersPage_);
    OfferReqBus_.RegisterCounters(CountersPage_);
    WarmerStateBusWriter_.RegisterCounters(CountersPage_);
    WarmerStateBusReader_.RegisterCounters(CountersPage_);
    Greylist_.RegisterCounters(CountersPage_, "Greylist");
    Sender_.RegisterCounters(CountersPage_);
    Boiler_.RegisterCounters(CountersPage_);
    GreyWarmer_.RegisterCounters(CountersPage_);
    RestartDetector_.RegisterCounters(CountersPage_);
    YtConfigPartners_.RegisterCounters(CountersPage_);
    CacheInvalidationService_.RegisterCounters(CountersPage_);
    TScheduler::Instance().RegisterCounters(CountersPage_);

    OfferBus_.Ignore(ru::yandex::travel::hotels::TPingMessage());
    OfferBus_.RegisterConverter(ru::yandex::travel::hotels::TSearcherMessage(), this, &TService::ConvertSearcherMessage);
    OfferBus_.Subscribe(NTravelProto::NBoiler::TSearcherMessageShort(), this, &TService::ProcessSearcherMessageShort);
    OfferBus_.Subscribe(NTravelProto::NOfferBus::TOfferInvalidationMessage(), [this](const TYtQueueMessage& busMessage) {
        NTravelProto::NOfferBus::TOfferInvalidationMessage message;
        if (!message.ParseFromString(busMessage.Bytes)) {
            throw yexception() << "Failed to parse TCacheInvalidationMessage record";
        }
        return CacheInvalidationService_.ProcessCacheInvalidationMessage(message);
    });
    OfferBus_.Ignore(NTravelProto::TTravellineCacheEvent());

    OfferReqBus_.Ignore(ru::yandex::travel::hotels::TPingMessage());
    OfferReqBus_.Subscribe(ru::yandex::travel::hotels::TOfferCacheReq(), this, &TService::ProcessOfferReqMessage);

    WarmerStateBusReader_.Subscribe(ru::yandex::travel::hotels::TGreyWarmerKeyBeginWarmMessage(), this, &TService::ProcessKeyBeginWarmMessage);
    WarmerStateBusReader_.Subscribe(ru::yandex::travel::hotels::TGreyWarmerKeyOffersFoundMessage(), this, &TService::ProcessKeyOffersFoundMessage);

    auto onBusReady = [this]() {
        with_lock (Lock_) {
            ++ReadyBusCount_;
            if (ReadyBusCount_ == 3) {
                Boiler_.OnInitialBusReadDone();
                GreyWarmer_.OnInitialBusReadDone();
            }
        }
    };
    OfferBus_.SetReadinessNotifier(onBusReady);
    OfferReqBus_.SetReadinessNotifier(onBusReady);
    WarmerStateBusReader_.SetReadinessNotifier(onBusReady);
    YtConfigPartners_.SetOnUpdateHandler([this](bool first) {
        Y_UNUSED(first);
        auto partners = YtConfigPartners_.GetAll();
        Boiler_.OnUpdatePartnersConfig(*(partners.get()));
        GreyWarmer_.OnUpdatePartnersConfig(*(partners.get()));
    });
}

void TService::Run() {
    TScheduler::Instance().Start();
    MonWebService_.Start(Config_.GetOther().GetMonitoringPort());
    Http_.Start(Config_.GetHttp());
    Greylist_.Start();
    CacheInvalidationService_.Start();
    OfferBus_.Start();
    OfferReqBus_.Start();
    YtConfigPartners_.Start();
    Sender_.Start();
    Boiler_.Start();
    GreyWarmer_.Start();
    WarmerStateBusWriter_.Start();
    WarmerStateBusReader_.Start();

    Stop_.WaitI();

    RestartDetector_.ReportShutdown();
    INFO_LOG << "Stopping Scheduler..." << Endl;
    TScheduler::Instance().Stop();
    INFO_LOG << "Stopping WarmerStateBusReader..." << Endl;
    WarmerStateBusReader_.Stop();
    INFO_LOG << "Stopping WarmerStateBusWriter..." << Endl;
    WarmerStateBusWriter_.Stop();
    INFO_LOG << "Stopping GreyWarmer..." << Endl;
    GreyWarmer_.Stop();
    INFO_LOG << "Stopping CacheInvalidationService..." << Endl;
    CacheInvalidationService_.Stop();
    INFO_LOG << "Stopping Boiler..." << Endl;
    Boiler_.Stop();
    INFO_LOG << "Stopping Sender..." << Endl;
    Sender_.Stop();
    INFO_LOG << "Stopping YtConfigPartners..." << Endl;
    YtConfigPartners_.Stop();
    INFO_LOG << "Stopping OfferReqBus..." << Endl;
    OfferReqBus_.Stop();
    INFO_LOG << "Stopping OfferBus..." << Endl;
    OfferBus_.Stop();
    INFO_LOG << "Stopping Greylist..." << Endl;
    Greylist_.Stop();
    INFO_LOG << "Stopping Http..." << Endl;
    Http_.Stop();
    INFO_LOG << "Stopping MonWebService..." << Endl;
    MonWebService_.Stop();
}

void TService::Stop() {
    Stop_.Signal();
}

TService::~TService() {
}

void TService::OnPingNanny(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    responseCb(THttpResponse(IsReady() ? HTTP_OK : HTTP_BAD_GATEWAY));
}

void TService::OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    if (auto v = httpReq.Query().Get("level")) {
        auto p = FromString<ELogPriority>(v);
        DoInitGlobalLog(CreateLogBackend(Config_.GetOther().GetMainLogFile(), p, true));
        responseCb(NHttp::TResponse::CreateText("Log level was changed to " + ToString(v) + "\n"));
    } else {
        responseCb(THttpResponse(HTTP_BAD_REQUEST));
    }
}

bool TService::IsReady() const {
    return Boiler_.IsReady() && GreyWarmer_.IsReady() && Greylist_.IsReady() && YtConfigPartners_.IsReady() && CacheInvalidationService_.IsReady();
}

void TService::ConvertSearcherMessage(TYtQueueMessage& busMessage) {
    ru::yandex::travel::hotels::TSearcherMessage msg;
    if (!msg.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TSearcherMessage record";
    }
    NTravelProto::NBoiler::TSearcherMessageShort msgShort;

    const auto& req = msg.GetRequest();
    auto* reqShort = msgShort.MutableRequest();
    reqShort->MutableHotelId()->CopyFrom(req.GetHotelId());
    reqShort->SetCheckInDate(req.GetCheckInDate());
    reqShort->SetCheckOutDate(req.GetCheckOutDate());
    reqShort->SetOccupancy(req.GetOccupancy());
    reqShort->SetCurrency(req.GetCurrency());
    reqShort->SetId(req.GetId());
    reqShort->SetPermalink(req.GetPermalink());
    reqShort->SetOfferCacheClientId(req.GetAttribution().GetOfferCacheClientId());

    const auto& resp = msg.GetResponse();
    auto* respShort = msgShort.MutableResponse();
    if (resp.HasError()) {
        respShort->MutableError()->CopyFrom(resp.GetError());
    } else if (resp.HasPlaceholder()) {
        respShort->MutablePlaceholder()->CopyFrom(resp.GetPlaceholder());
    } else if (resp.HasOffers()) {
        respShort->MutableOffersInfo()->SetHasOffers(!resp.GetOffers().GetOffer().empty());
    }

    busMessage.Bytes = msgShort.SerializeAsString();
    busMessage.MessageType = msgShort.GetDescriptor()->full_name();

}

bool TService::ProcessSearcherMessageShort(const TYtQueueMessage& busMessage) {
    NTravelProto::NBoiler::TSearcherMessageShort message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TSearcherMessageShort record";
    }
    const auto& req = message.GetRequest();
    const auto& resp = message.GetResponse();

    if (resp.HasError()) {
        // Ошибка при исполнении запроса, но исполнение закончено, данных не получено
        TInstant expireTimestamp = busMessage.ExpireTimestamp;
        const auto& error = resp.GetError();
        DEBUG_LOG << "Searcher responded with an error (RequestId = " << req.GetId()
                  << ", OfferCacheClientId = " << req.GetOfferCacheClientId()  << "): "
                  << NTravelProto::EErrorCode_Name(error.GetCode()) << ", " << error.GetMessage() << Endl;
        TProfileTimer timer;
        Boiler_.OnRequestFinished(req, busMessage.Timestamp, expireTimestamp, true);
        Counters_.OfferBusProcessBoilerMicros += timer.Step().MicroSeconds();
        GreyWarmer_.OnRequestFinished(req, false, true);
        Counters_.OfferBusProcessGreyWarmerMicros += timer.Step().MicroSeconds();
    } else if (resp.HasPlaceholder()) {
        return false; // Запрос начался, данных еще нет, это нам совсем без интереса
    } else if (resp.HasOffersInfo()) {
        TInstant expireTimestamp = busMessage.ExpireTimestamp - TDuration::Seconds(Config_.GetCommon().GetKeyFinishedExpireRelativeTimeoutSec());
        bool hasOffers = resp.GetOffersInfo().GetHasOffers();
        TProfileTimer timer;
        Boiler_.OnRequestFinished(req, busMessage.Timestamp, expireTimestamp, false);
        Counters_.OfferBusProcessBoilerMicros += timer.Step().MicroSeconds();
        GreyWarmer_.OnRequestFinished(req, hasOffers, false);
        Counters_.OfferBusProcessGreyWarmerMicros += timer.Step().MicroSeconds();
    }
    return true;
}

bool TService::ProcessOfferReqMessage(const TYtQueueMessage& busMessage) {
    ru::yandex::travel::hotels::TOfferCacheReq message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TOfferCacheReq record";
    }
    TProfileTimer timer;
    Boiler_.OnCacheKeyUse(message, busMessage.Timestamp);
    Counters_.OfferReqBusProcessBoilerMicros += timer.Step().MicroSeconds();
    return true;
}

bool TService::ProcessKeyBeginWarmMessage(const TYtQueueMessage& busMessage) {
    ru::yandex::travel::hotels::TGreyWarmerKeyBeginWarmMessage message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TGreyWarmerKeyBeginWarmMessage record";
    }
    TProfileTimer timer;
    GreyWarmer_.OnBeginWarmMessage(message, busMessage.Timestamp);
    Counters_.WarmerBusProcessMicros += timer.Step().MicroSeconds();
    return true;
}

bool TService::ProcessKeyOffersFoundMessage(const TYtQueueMessage& busMessage) {
    ru::yandex::travel::hotels::TGreyWarmerKeyOffersFoundMessage message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TGreyWarmerKeyOffersFoundMessage record";
    }
    TProfileTimer timer;
    GreyWarmer_.OnOffersFoundMessage(message, busMessage.Timestamp);
    Counters_.WarmerBusProcessMicros += timer.Step().MicroSeconds();
    return true;
}

} // namespace NTravel
