#include "service.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/proto/offerbus_messages/bus_messages.pb.h>

#include <library/cpp/json/json_writer.h>

namespace NTravel::NBusBroker {
    TService::TService(const NTravelProto::NBusBroker::TConfig& config)
        : Config(config)
        , RestartDetector(config.GetOther().GetRestartDetectorStateFile())
        , TvmService(config.GetTvmService())
        , OfferBusWriter(Config.GetOfferBusWriter(), "OfferBusWriter")
        , YtConfigPartners("YtConfigPartners", config.GetYtConfigPartners())
        , PermalinkToOriginalIdsMapper("PermalinkToOriginalIdsMapper", Config.GetPermalinkToOriginalIdsMapper())
        , PermalinkToClusterMapper("PermalinkToClusterMapper", Config.GetPermalinkToClusterMapper())
        , GrpcServer(NGrpc::TAsyncServerConfig(config.GetGrpcServer().GetBindAddress(), Config.GetGrpcServer().GetReplyThreads()))
        , Counters(*this)
    {
        CountersPage.AddToHttpService(MonWebService);

        CountersPage.RegisterSource(&Counters, "Service");
        YtConfigPartners.RegisterCounters(CountersPage);
        RestartDetector.RegisterCounters(CountersPage);
        TvmService.RegisterCounters(CountersPage);
        OfferBusWriter.RegisterCounters(CountersPage);
        PermalinkToOriginalIdsMapper.RegisterCounters(CountersPage);
        PermalinkToClusterMapper.RegisterCounters(CountersPage);
        Http.RegisterCounters(CountersPage);
        GrpcServer.RegisterCounters(CountersPage, "GrpcServer");

        Http.SetTvm(&TvmService);
        GrpcServer.SetTvm(&TvmService);

        Http.AddHandler("/setlog", NHttp::Local(), this, &TService::OnSetLogLevel);
        Http.AddHandler("/shutdown", NHttp::Local(), this, &TService::OnShutdown);
        Http.AddHandler("/ping", NHttp::ExternalWithoutTvm(), this, &TService::OnPing);
        Http.AddHandler("/invalidate_offers", NHttp::ExternalWithTvm(), this, &TService::OnHttpInvalidateOffers);

        REGISTER_GRPC_HANDLER(GrpcServer, InvalidateOffers, this, &TService::OnGrpcInvalidateOffers);
        REGISTER_GRPC_HANDLER(GrpcServer, Ping, this, &TService::OnGrpcPing);

        auto permalinkMapper = [this](NPermalinkMappers::TPermalinkToOriginalIdsMapper::TKey* key) {
            *key = PermalinkToClusterMapper.GetClusterPermalink(*key);
        };
        PermalinkToOriginalIdsMapper.SetMappingFilters(permalinkMapper, NPermalinkMappers::TPermalinkToOriginalIdsMappingRecJoiner::Join);

        PermalinkToClusterMapper.SetOnFinishHandler([this]() {
            PermalinkToOriginalIdsMapper.Reload();
        });
        PermalinkToClusterMapper.Start();

        YtConfigPartners.SetOnUpdateHandler([this](bool first) {
            auto partners = YtConfigPartners.GetAll();
            THashMap<TString, EPartnerId> partnerIdByCode;
            partnerIdByCode.clear();
            for (const auto& [id, partner] : *partners) {
                partnerIdByCode[partner.GetCode()] = id;
            }
            PermalinkToOriginalIdsMapper.SetPartnerIdByCode(partnerIdByCode);
            if (first) {
                PermalinkToOriginalIdsMapper.Start();
            }
        });
    }

    void TService::Start() {
        TScheduler::Instance().Start();
        MonWebService.Start(Config.GetOther().GetMonitoringPort());
        YtConfigPartners.Start();
        OfferBusWriter.Start();
        GrpcServer.Start();
        Http.Start(Config.GetHttp());
    }

    void TService::Wait() {
        StopEvent.WaitI();
    }

    void TService::Stop() {
        Http.Stop();
        GrpcServer.Stop();
        OfferBusWriter.Stop();
        YtConfigPartners.Stop();
        PermalinkToOriginalIdsMapper.Stop();
        PermalinkToClusterMapper.Stop();
        MonWebService.Stop();
        RestartDetector.ReportShutdown();

        StopEvent.Signal();
    }

    void TService::OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        if (auto v = httpReq.Query().Get("level")) {
            auto p = FromString<ELogPriority>(v);
            DoInitGlobalLog(CreateLogBackend(Config.GetOther().GetMainLogFile(), p, true));
            responseCb(NHttp::TResponse::CreateText("Log level was changed to " + ToString(v) + "\n"));
        } else {
            responseCb(THttpResponse(HTTP_BAD_REQUEST));
        }
    }

    void TService::OnPing(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        Y_UNUSED(httpReq);
        bool isReady = IsReady();
        responseCb(NHttp::TResponse::CreateText(isReady ? "OK" : "Not ready", isReady ? HTTP_OK : HTTP_BAD_GATEWAY));
    }

    void TService::OnShutdown(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        Y_UNUSED(httpReq);
        if (!IsShuttingDown.TrySet()) {
            responseCb(NHttp::TResponse::CreateText("Already shutting down\n"));
            return;
        }
        INFO_LOG << "Shutting down!" << Endl;
        SystemThreadFactory()->Run([this]() {
            Http.Shutdown();
            GrpcServer.Shutdown();
            auto dur = TDuration::Seconds(7);
            INFO_LOG << "Http & gRPC Shutdown called, wait " << dur << Endl;
            StopEvent.WaitT(dur);
            Stop();
        });
        responseCb(NHttp::TResponse::CreateText("Shutdown procedure started\n"));
    }

    void TService::OnHttpInvalidateOffers(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        NTravelProto::NOfferInvalidation::TOfferInvalidationReq req;
        try {
            NTravel::NProtobuf::ParseCgiRequest(httpReq.Query(), &req);
        } catch (...) {
            WARNING_LOG << "Bad http invalidate offers request: " << CurrentExceptionMessage() << Endl;
            Counters.NHttpBadRequests.Inc();
            responseCb(NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST));
            return;
        }
        try {
            InvalidateOffers(req);
            auto resp = NTravelProto::NOfferInvalidation::TOfferInvalidationResp();
            resp.MutableResult();
            resp.MutableError()->SetMessage(CurrentExceptionMessage());
            responseCb(NHttp::TResponse::CreateJson(resp, HTTP_OK, true));
        } catch (const TBadRequestException& e) {
            WARNING_LOG << "Bad http invalidate offers request: " << CurrentExceptionMessage() << Endl;
            Counters.NHttpBadRequests.Inc();
            responseCb(NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST));
            return;
        } catch (...) {
            ERROR_LOG << "Error while handling http invalidate request: " << CurrentExceptionMessage() << Endl;
            Counters.NHttpUnexpectedErrors.Inc();
            responseCb(NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_INTERNAL_SERVER_ERROR));
            return;
        }
    }

    void TService::OnGrpcInvalidateOffers(const NTravelProto::NOfferInvalidation::TOfferInvalidationReq& req,
                                          const NGrpc::TServerReqMetadata& srvMeta,
                                          const TGrpcServer::TResponseCb<NTravelProto::NOfferInvalidation::TOfferInvalidationResp>& responseCb) {
        DEBUG_LOG << "Got gRPC request callId " << srvMeta.CallId << " from '" << srvMeta.RemoteFQDN << Endl;
        try {
            InvalidateOffers(req);
            auto resp = NTravelProto::NOfferInvalidation::TOfferInvalidationResp();
            resp.MutableResult();
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
        } catch (const TBadRequestException& e) {
            auto resp = NTravelProto::NOfferInvalidation::TOfferInvalidationResp();
            resp.MutableError()->SetMessage(CurrentExceptionMessage());
            WARNING_LOG << "Bad grpc invalidate offers request: " << CurrentExceptionMessage() << Endl;
            Counters.NGrpcBadRequests.Inc();
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
            return;
        } catch (...) {
            auto resp = NTravelProto::NOfferInvalidation::TOfferInvalidationResp();
            resp.MutableError()->SetMessage(CurrentExceptionMessage());
            ERROR_LOG << "Error while handling grpc invalidate offers request: " << CurrentExceptionMessage() << Endl;
            Counters.NGrpcUnexpectedErrors.Inc();
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
            return;
        }
    }

    void TService::OnGrpcPing(const NTravelProto::TPingRpcReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TGrpcServer::TResponseCb<NTravelProto::TPingRpcRsp>& responseCb) {
        Y_UNUSED(req, srvMeta);
        NTravelProto::TPingRpcRsp resp;
        resp.SetIsReady(IsReady());
        responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
    }

    void TService::InvalidateOffers(const NTravelProto::NOfferInvalidation::TOfferInvalidationReq& req) {
        auto message = NTravelProto::NOfferBus::TOfferInvalidationMessage();
        if (req.GetCurrency() == NTravelProto::ECurrency::C_UNKNOWN) {
            throw TBadRequestException() << "Request with unknown Currency";
        }
        if (!req.HasTimestamp()) {
            throw TBadRequestException() << "Request without Timestamp";
        }
        if (req.FiltersSize() == 0) {
            throw TBadRequestException() << "Request without Filters";
        }
        if (req.GetOfferInvalidationSource() == NTravelProto::NOfferInvalidation::EOfferInvalidationSource::OIS_UNKNOWN) {
            throw TBadRequestException() << "Request with unknown OfferInvalidationSource";
        }
        if (req.HasHotelId()) {
            auto event = message.MutableEvent();
            FillEventFields(req.GetHotelId(), req, event);
            OfferBusWriter.Write(message);
        } else if (req.HasPermalink()) {
            TPermalink clusterPermalink = PermalinkToClusterMapper.GetClusterPermalink(req.GetPermalink());
            if (auto partnerHotelIds = PermalinkToOriginalIdsMapper.GetMapping(clusterPermalink)) {
                for (const auto& partnerHotelId : partnerHotelIds->PartnerIds) {
                    auto event = message.MutableEvent();
                    auto pbHotelId = NTravelProto::THotelId();
                    partnerHotelId.ToProto(&pbHotelId);
                    FillEventFields(pbHotelId, req, event);
                    OfferBusWriter.Write(message);
                }
            }
        } else {
            throw TBadRequestException() << "Request has no hotelId neither permalink";
        }
    }

    void TService::FillEventFields(const NTravelProto::THotelId& hotelId,
                                   const NTravelProto::NOfferInvalidation::TOfferInvalidationReq& req,
                                   NTravelProto::NOfferInvalidation::TOfferInvalidationEvent* event) const {
        event->MutableHotelId()->CopyFrom(hotelId);
        event->SetCurrency(req.GetCurrency());
        event->MutableTimestamp()->CopyFrom(req.GetTimestamp());
        event->MutableFilters()->CopyFrom(req.GetFilters());
        event->SetOfferInvalidationSource(req.GetOfferInvalidationSource());
    }

    bool TService::IsReady() const {
        return YtConfigPartners.IsReady() && PermalinkToClusterMapper.IsReady() && PermalinkToOriginalIdsMapper.IsReady();
    }

    TService::TCounters::TCounters(const TService& service)
        : Service(service)
    {
    }

    void TService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        IsReady = Service.IsReady() ? 1 : 0;

        ct->insert(MAKE_COUNTER_PAIR(IsReady));

        ct->insert(MAKE_COUNTER_PAIR(NHttpBadRequests));
        ct->insert(MAKE_COUNTER_PAIR(NHttpUnexpectedErrors));

        ct->insert(MAKE_COUNTER_PAIR(NGrpcBadRequests));
        ct->insert(MAKE_COUNTER_PAIR(NGrpcUnexpectedErrors));
    }
}
