# -*- coding: utf-8 -*-
from __future__ import annotations

from dataclasses import dataclass, field
from typing import Dict, List, Optional, Type

from travel.hotels.content_manager.data_model.storage import EntityClass, EntityClassWithStatus
from travel.hotels.content_manager.data_model.storage import (
    StorageHotelWL, StorageMapping, StoragePermalink, StoragePermalinkWL, StoragePermaroom, StorageSCDescription,
    StorageUrl
)
from travel.hotels.content_manager.data_model.types import ProcessType
from travel.hotels.content_manager.lib.processor import Processor
from travel.hotels.content_manager.lib.trigger import FilterCallable, Producer
from travel.hotels.content_manager.lib.updater import Updater, StubUpdater
from travel.hotels.content_manager.stages.actualization.mock import MockActualization
from travel.hotels.content_manager.stages.actualization.trigger import FILTERS_ACTUALIZATION, ProducerActualization
from travel.hotels.content_manager.stages.actualization.updater import UpdaterActualization
from travel.hotels.content_manager.stages.call_center.mock import MockCallCenter
from travel.hotels.content_manager.stages.call_center.processor import ProcessorCallCenter
from travel.hotels.content_manager.stages.call_center.trigger import ProducerCallCenter
from travel.hotels.content_manager.stages.call_center.updater import UpdaterCallCenter
from travel.hotels.content_manager.stages.catroom_export.processor import ProcessorCatroomExport
from travel.hotels.content_manager.stages.clusterization.mock import MockClusterization
from travel.hotels.content_manager.stages.clusterization.trigger import FILTERS_CLUSTERIZATION, ProducerClusterization
from travel.hotels.content_manager.stages.clusterization.updater import UpdaterClusterization
from travel.hotels.content_manager.stages.cr_known_permalinks.processor import ProcessorCrKnownPermalinks
from travel.hotels.content_manager.stages.cr_known_permalinks.updater import UpdaterCrKnownPermalinks
from travel.hotels.content_manager.stages.offer_prioritization.mock import MockOfferPrioritization
from travel.hotels.content_manager.stages.offer_prioritization.processor import ProcessorOfferPrioritization
from travel.hotels.content_manager.stages.offer_prioritization.trigger import ProducerOfferPrioritization
from travel.hotels.content_manager.stages.offer_prioritization.updater import UpdaterOfferPrioritization
from travel.hotels.content_manager.stages.partner_offers.mock import MockPartnerOffers
from travel.hotels.content_manager.stages.partner_offers.processor import ProcessorPartnerOffers
from travel.hotels.content_manager.stages.partner_offers.trigger import ProducerPartnerOffers
from travel.hotels.content_manager.stages.partner_offers.updater import UpdaterPartnerOffers
from travel.hotels.content_manager.stages.sc_export_descriptions.processor import ProcessorSCExportDescriptions
from travel.hotels.content_manager.stages.sc_new_descriptions.processor import ProcessorSCNewDescriptions
from travel.hotels.content_manager.stages.sc_new_descriptions.updater import UpdaterSCNewDescriptions
from travel.hotels.content_manager.stages.sc_update_descriptions.mock import MockSCUpdateDescriptions
from travel.hotels.content_manager.stages.sc_update_descriptions.trigger import ProducerSCUpdateDescriptions
from travel.hotels.content_manager.stages.sc_update_descriptions.updater import UpdaterSCUpdateDescriptions
from travel.hotels.content_manager.stages.top_permalinks.processor import ProcessorTopPermalinks
from travel.hotels.content_manager.stages.top_permalinks.updater import UpdaterTopPermalinks
from travel.hotels.content_manager.stages.update_mappings.mock import MockUpdateMappings
from travel.hotels.content_manager.stages.update_mappings.processor import ProcessorUpdateMappings
from travel.hotels.content_manager.stages.update_mappings.trigger import ProducerUpdateMappings
from travel.hotels.content_manager.stages.update_mappings.updater import UpdaterUpdateMappings
from travel.hotels.content_manager.stages.wl_clusterized_hotels.mock import MockWLClusterizedHotels
from travel.hotels.content_manager.stages.wl_clusterized_hotels.processor import ProcessorWLClusterizedHotels
from travel.hotels.content_manager.stages.wl_clusterized_hotels.trigger import ProducerWLClusterizedHotels
from travel.hotels.content_manager.stages.wl_clusterized_hotels.updater import UpdaterWLClusterizedHotels
from travel.hotels.content_manager.stages.wl_export.processor import ProcessorWLExport
from travel.hotels.content_manager.stages.wl_get_hotel_info.mock import MockWLGetHotelInfo
from travel.hotels.content_manager.stages.wl_get_hotel_info.processor import ProcessorWLGetHotelInfo
from travel.hotels.content_manager.stages.wl_get_hotel_info.trigger import ProducerWLGetHotelInfo
from travel.hotels.content_manager.stages.wl_get_hotel_info.updater import UpdaterWLGetHotelInfo
from travel.hotels.content_manager.stages.wl_match_hotels.mock import MockWLMatchHotels
from travel.hotels.content_manager.stages.wl_match_hotels.trigger import ProducerWLMatchHotels
from travel.hotels.content_manager.stages.wl_match_hotels.updater import UpdaterWLMatchHotels
from travel.hotels.content_manager.stages.wl_new_hotels.mock import MockWLNewHotels
from travel.hotels.content_manager.stages.wl_new_hotels.processor import ProcessorWLNewHotels
from travel.hotels.content_manager.stages.wl_new_hotels.updater import UpdaterWLNewHotels
from travel.hotels.content_manager.stages.wl_start.processor import ProcessorWLStart
from travel.hotels.content_manager.stages.wl_start.updater import UpdaterWLStart
from travel.hotels.content_manager.stages.yang_mappings.mock import MockMonkeyMappings
from travel.hotels.content_manager.stages.yang_mappings.trigger import ProducerYangMappings
from travel.hotels.content_manager.stages.yang_mappings.updater import UpdaterYangMappings
from travel.hotels.content_manager.stages.yang_rooms.mock import MockMonkeyRooms
from travel.hotels.content_manager.stages.yang_rooms.trigger import FILTERS_YANG_ROOMS, ProducerYangRooms
from travel.hotels.content_manager.stages.yang_rooms.updater import UpdaterYangRooms


@dataclass(frozen=True)
class StageConfig:
    process_type: ProcessType
    name: str
    producer: Type[Producer] = None
    processor: Type[Processor] = None
    mock: Type[Processor] = None
    updater: Type[Updater] = None
    jobs_max: Optional[int] = None
    job_size: Optional[int] = None
    manual_start: bool = False
    run_on_storage_change: bool = False
    entity_cls: Optional[EntityClassWithStatus] = None
    other_entities: List[EntityClass] = field(default_factory=list)
    filters: Optional[Dict[str, FilterCallable]] = None
    save_history: bool = True
    job_retry_count: Optional[int] = None
    job_max_run_time: int = 5 * 60 * 60


TopPermalinksStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='top_permalinks',
    processor=ProcessorTopPermalinks,
    updater=UpdaterTopPermalinks,
    jobs_max=1,
    manual_start=True,
)


PartnerOffersStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='partner_offers',
    producer=ProducerPartnerOffers,
    processor=ProcessorPartnerOffers,
    mock=MockPartnerOffers,
    updater=UpdaterPartnerOffers,
    entity_cls=StoragePermalink,
)


YangRoomsStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='yang_rooms',
    producer=ProducerYangRooms,
    mock=MockMonkeyRooms,
    updater=UpdaterYangRooms,
    entity_cls=StoragePermalink,
    other_entities=[StoragePermaroom, StorageMapping, StorageUrl],
    filters=FILTERS_YANG_ROOMS,
)


CrKnownPermalinksStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='cr_known_permalinks',
    processor=ProcessorCrKnownPermalinks,
    updater=UpdaterCrKnownPermalinks,
    jobs_max=1,
    manual_start=True,
)


UpdateMappingsStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='update_mappings',
    producer=ProducerUpdateMappings,
    processor=ProcessorUpdateMappings,
    mock=MockUpdateMappings,
    updater=UpdaterUpdateMappings,
    entity_cls=StoragePermalink,
)


OfferPrioritizationStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='offer_prioritization',
    producer=ProducerOfferPrioritization,
    processor=ProcessorOfferPrioritization,
    mock=MockOfferPrioritization,
    updater=UpdaterOfferPrioritization,
    jobs_max=1,
    entity_cls=StorageMapping,
    other_entities=[StoragePermalink],
)


YangMappingsStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='yang_mappings',
    producer=ProducerYangMappings,
    mock=MockMonkeyMappings,
    updater=UpdaterYangMappings,
    entity_cls=StorageMapping,
    other_entities=[StoragePermalink, StoragePermaroom],
)


CatroomExportStageConfig = StageConfig(
    process_type=ProcessType.CATROOM,
    name='catroom_export',
    processor=ProcessorCatroomExport,
    updater=StubUpdater,
    run_on_storage_change=True,
    save_history=False,
)


WLStartStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='wl_start',
    processor=ProcessorWLStart,
    updater=UpdaterWLStart,
    jobs_max=5,
    manual_start=True,
)


ActualizationStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='actualization',
    producer=ProducerActualization,
    mock=MockActualization,
    updater=UpdaterActualization,
    jobs_max=3,
    job_size=50,
    entity_cls=StoragePermalinkWL,
    filters=FILTERS_ACTUALIZATION,
)


CallCenterStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='call_center',
    producer=ProducerCallCenter,
    processor=ProcessorCallCenter,
    updater=UpdaterCallCenter,
    mock=MockCallCenter,
    jobs_max=2,
    job_size=50,
    entity_cls=StoragePermalinkWL,
)


ClusterizationStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='clusterization',
    producer=ProducerClusterization,
    mock=MockClusterization,
    updater=UpdaterClusterization,
    jobs_max=3,
    job_size=50,
    entity_cls=StoragePermalinkWL,
    filters=FILTERS_CLUSTERIZATION,
)


WLNewHotelsStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='wl_new_hotels',
    processor=ProcessorWLNewHotels,
    mock=MockWLNewHotels,
    updater=UpdaterWLNewHotels,
    jobs_max=1,
    manual_start=True,
)


WLGetHotelInfoStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='wl_get_hotel_info',
    producer=ProducerWLGetHotelInfo,
    processor=ProcessorWLGetHotelInfo,
    mock=MockWLGetHotelInfo,
    updater=UpdaterWLGetHotelInfo,
    entity_cls=StorageHotelWL,
    other_entities=[StoragePermalinkWL],
)


WLMatchHotelsStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='wl_match_hotels',
    producer=ProducerWLMatchHotels,
    mock=MockWLMatchHotels,
    updater=UpdaterWLMatchHotels,
    job_size=200,
    entity_cls=StorageHotelWL,
    other_entities=[StoragePermalinkWL],
)


WLClusterizedHotelsStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='wl_clusterized_hotels',
    producer=ProducerWLClusterizedHotels,
    processor=ProcessorWLClusterizedHotels,
    mock=MockWLClusterizedHotels,
    updater=UpdaterWLClusterizedHotels,
    entity_cls=StoragePermalinkWL,
    job_retry_count=3,
)


WLExportStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='wl_export',
    processor=ProcessorWLExport,
    updater=StubUpdater,
    run_on_storage_change=True,
    save_history=False,
)


SCNewDescriptionsStageConfig = StageConfig(
    process_type=ProcessType.SERVICE_CLASS,
    name='sc_new_descriptions',
    processor=ProcessorSCNewDescriptions,
    updater=UpdaterSCNewDescriptions,
    jobs_max=1,
    manual_start=True,
)


SCUpdateDescriptionsStageConfig = StageConfig(
    process_type=ProcessType.SERVICE_CLASS,
    name='sc_update_descriptions',
    producer=ProducerSCUpdateDescriptions,
    mock=MockSCUpdateDescriptions,
    updater=UpdaterSCUpdateDescriptions,
    entity_cls=StorageSCDescription,
)


SCExportDescriptionsStageConfig = StageConfig(
    process_type=ProcessType.WHITELIST,
    name='sc_export_descriptions',
    processor=ProcessorSCExportDescriptions,
    updater=StubUpdater,
    run_on_storage_change=True,
    save_history=False,
)


STAGE_LIST = [
    CatroomExportStageConfig,
    ActualizationStageConfig,
    ClusterizationStageConfig,
    CallCenterStageConfig,
    TopPermalinksStageConfig,
    PartnerOffersStageConfig,
    CrKnownPermalinksStageConfig,
    UpdateMappingsStageConfig,
    OfferPrioritizationStageConfig,
    YangRoomsStageConfig,
    YangMappingsStageConfig,
    WLStartStageConfig,
    WLNewHotelsStageConfig,
    WLGetHotelInfoStageConfig,
    WLMatchHotelsStageConfig,
    WLClusterizedHotelsStageConfig,
    WLExportStageConfig,
    SCNewDescriptionsStageConfig,
    SCUpdateDescriptionsStageConfig,
    SCExportDescriptionsStageConfig,
]
