# -*- coding: utf-8 -*-

# !!! DO NOT USE "from __future__ import annotations" here
# This will break deserialization from dict

from dataclasses import dataclass, field
from enum import Enum
from typing import Dict, List, Optional, Type, Union

from travel.hotels.content_manager.data_model.common import HotelInfo
from travel.hotels.content_manager.data_model.types import (
    ActualizationStartReason, AssigneeSkill, ClusterizationStartReason, DEFAULT_ENTITY_PRIORITY, StageResult,
    SCDescriptionResult, WLResult, uint
)


PERMAROOM_ID_OFFSET = 1_000_000


class StageStatus(Enum):
    NOTHING_TO_DO = 'nothing_to_do'
    TO_BE_PROCESSED = 'to_be_processed'
    IN_PROCESS = 'in_process'


@dataclass
class StoragePermalink:
    id: uint = field(default=0, metadata={'is_key': True})
    is_deleted: bool = False
    hotel_url: str = ''
    marked_by: str = ''
    show_other: bool = True
    original_permalink: uint = 0
    grouping_key: str = ''
    priority: uint = DEFAULT_ENTITY_PRIORITY
    status_partner_offers: StageStatus = StageStatus.NOTHING_TO_DO
    status_partner_offers_ts: uint = 0
    status_yang_rooms: StageStatus = StageStatus.NOTHING_TO_DO
    status_yang_rooms_ts: uint = 0
    status_update_mappings: StageStatus = StageStatus.NOTHING_TO_DO
    status_update_mappings_ts: uint = 0


@dataclass
class StorageHotelWL(HotelInfo):
    permalink: uint = field(default=0, metadata={'is_key': True})
    partner_id: str = field(default='', metadata={'is_key': True})
    original_id: str = field(default='', metadata={'is_key': True})
    is_wl_approved: bool = False
    grouping_key: str = ''
    priority: uint = DEFAULT_ENTITY_PRIORITY
    status_wl_get_hotel_info: StageStatus = StageStatus.NOTHING_TO_DO
    status_wl_get_hotel_info_ts: uint = 0
    status_wl_match_hotels: StageStatus = StageStatus.NOTHING_TO_DO
    status_wl_match_hotels_ts: uint = 0


@dataclass
class StoragePermalinkWL(HotelInfo):
    permalink: uint = field(default=0, metadata={'is_key': True})
    grouping_key: str = ''
    priority: uint = DEFAULT_ENTITY_PRIORITY
    requirements: str = ''
    date_next_check: str = ''
    required_stages: str = ''
    finished_stages: str = ''
    route: str = ''
    hotel_name: str = ''
    assignee_skill: AssigneeSkill = AssigneeSkill.BASIC
    required_attributes: str = ''
    checked_attributes: str = ''
    comments: str = ''
    # actualization
    actualization_start_reason: ActualizationStartReason = ActualizationStartReason.MAIN_ATTRIBUTE_ACTUALIZATION
    actualization_iteration: uint = 1
    actualization_required_stages: str = ''
    actualization_result: StageResult = StageResult.UNKNOWN
    status_actualization: StageStatus = StageStatus.NOTHING_TO_DO
    status_actualization_ts: uint = 0
    # call_center
    call_center_required_stages: str = ''
    call_center_result: StageResult = StageResult.UNKNOWN
    status_call_center: StageStatus = StageStatus.NOTHING_TO_DO
    status_call_center_ts: uint = 0
    call_center_request_id: uint = None
    call_center_iteration: uint = 0
    # clusterization
    clusterization_start_reason: ClusterizationStartReason = ClusterizationStartReason.COMMON
    clusterization_required_stages: str = ''
    clusterization_iteration: uint = 1
    clusterization_result: StageResult = StageResult.UNKNOWN
    status_clusterization: StageStatus = StageStatus.NOTHING_TO_DO
    status_clusterization_ts: uint = 0
    # wl_clusterized_hotels
    status_wl_clusterized_hotels: StageStatus = StageStatus.NOTHING_TO_DO
    status_wl_clusterized_hotels_ts: uint = 0
    wl_result: WLResult = WLResult.UNKNOWN


@dataclass
class StoragePermaroom:
    id: int = field(default=None, metadata={'is_key': True})
    name: str = ''
    is_deleted: bool = False
    permalink: uint = field(default=0, metadata={'is_key': True, 'foreign_key': StoragePermalink.id})
    comment: str = ''
    alternative_names: str = ''


@dataclass
class StorageMapping:
    is_deleted: bool = False
    is_hidden: bool = False
    is_banned: bool = False
    need_new_permaroom: bool = False
    permalink: uint = field(default=0, metadata={'is_key': True, 'foreign_key': StoragePermalink.id})
    operator_id: str = field(default='', metadata={'is_key': True})
    orig_hotel_id: str = field(default='', metadata={'is_key': True})
    mapping_key: str = field(default='', metadata={'is_key': True})
    permaroom_id: Optional[int] = field(default=None, metadata={'foreign_key': StoragePermaroom.id})
    orig_room_name: str = ''
    count: int = 0
    prices_per_night: str = ''
    requests_range: str = ''
    price: int = 0
    url: str = ''
    params: str = ''
    comment: str = ''
    grouping_key: str = ''
    priority: uint = DEFAULT_ENTITY_PRIORITY
    status_yang_mappings: StageStatus = StageStatus.NOTHING_TO_DO
    status_yang_mappings_ts: uint = 0
    status_offer_prioritization: StageStatus = StageStatus.NOTHING_TO_DO
    status_offer_prioritization_ts: uint = 0
    counters: Dict[str, int] = field(default_factory=dict)


@dataclass
class StorageUrl:
    permalink: uint = field(default=0, metadata={'is_key': True, 'foreign_key': StoragePermalink.id})
    provider: str = field(default='', metadata={'is_key': True})
    url: str = field(default='', metadata={'is_key': True})


@dataclass
class StorageSCDescription:
    carrier_code: str = field(default='', metadata={'is_key': True})
    car_type_code: str = field(default='', metadata={'is_key': True})
    sc_code: str = field(default='', metadata={'is_key': True})
    data_source: str = ''
    sc_description: str = ''
    sc_description_original: str = ''
    sc_description_specific: List[Dict[str, str]] = field(default_factory=list)
    sc_description_result: SCDescriptionResult = SCDescriptionResult.UNKNOWN
    grouping_key: str = ''
    priority: uint = DEFAULT_ENTITY_PRIORITY
    status_sc_update_descriptions: StageStatus = StageStatus.NOTHING_TO_DO
    status_sc_update_descriptions_ts: uint = 0


TABLE_NAMES = {
    'permalinks': StoragePermalink,
    'permarooms': StoragePermaroom,
    'mappings': StorageMapping,
    'urls': StorageUrl,
    'hotels_wl': StorageHotelWL,
    'permalinks_wl': StoragePermalinkWL,
    'sc_descriptions': StorageSCDescription,
}


CatroomEntityClass = Union[Type[StoragePermalink], Type[StoragePermaroom], Type[StorageMapping], Type[StorageUrl]]
WhitelistEntityClass = Union[Type[StorageHotelWL], Type[StoragePermalinkWL]]
SCDescriptionEntityClass = Type[StorageSCDescription]
EntityClass = Union[CatroomEntityClass, WhitelistEntityClass, SCDescriptionEntityClass]


CatroomEntityClassWithStatus = Union[Type[StoragePermalink], Type[StorageMapping]]
WhitelistEntityClassWithStatus = Union[Type[StorageHotelWL], Type[StoragePermalinkWL]]
SCDescriptionEntityClassWithStatus = Type[StorageSCDescription]
EntityClassWithStatus = Union[
    CatroomEntityClassWithStatus,
    WhitelistEntityClassWithStatus,
    SCDescriptionEntityClassWithStatus,
]


CatroomEntityWithStatus = Union[StoragePermalink, StorageMapping]
WhitelistEntityWithStatus = Union[StorageHotelWL, StoragePermalinkWL]
SCDescriptionEntityWithStatus = StorageSCDescription
EntityWithStatus = Union[CatroomEntityWithStatus, WhitelistEntityWithStatus, SCDescriptionEntityWithStatus]


DispatchableEntity = StoragePermalinkWL
AdvanceableEntityCls = Type[StoragePermalinkWL]
