# -*- coding: utf-8 -*-
from __future__ import annotations

from abc import abstractmethod, ABCMeta
from argparse import Namespace
from typing import Optional
import logging

from travel.hotels.content_manager.lib.path_info import PathInfo
from travel.hotels.content_manager.lib.persistence_manager import PersistenceManager
from travel.hotels.content_manager.lib.yql_simple_client import YqlSimpleClient


LOG = logging.getLogger(__name__)


class Processor(object):
    __metaclass__ = ABCMeta

    def __init__(
            self,
            persistence_manager: PersistenceManager,
            yql_client: Optional[YqlSimpleClient],
            path_info: PathInfo,
            input_path: str,
            output_path: str,
            job_id: str,
            options: Optional[Namespace] = None,
    ) -> None:
        self.persistence_manager = persistence_manager
        self.yql_client = yql_client

        self.path_info = path_info
        self.input_path = input_path
        self.output_path = output_path
        self.job_id = job_id

        self.options = options

    @abstractmethod
    def run(self) -> None:
        pass

    def process(self) -> None:
        if not self.persistence_manager.exists(self.output_path):
            LOG.info(f'Creating output dir {self.output_path}')
            self.persistence_manager.create_dir(self.output_path)
        self.run()

    def post_process(self):
        # to run something out of process transaction
        pass
