from enum import Enum
from typing import Any

from travel.hotels.content_manager.lib.persistence_manager import PersistenceManager


class ProcessorRunProgress(Enum):
    ENQUEUED = 'enqueued'
    RUNNING = 'running'
    FINISHED = 'finished'


class ProcessorRunStatus(Enum):
    UNKNOWN = 'unknown'
    SUCCESS = 'success'
    FAILED = 'failed'


class ProcessorRunInfo:

    __processing_start_ts_attribute__ = 'processing_start_ts'
    __progress_attribute__ = 'run_progress'
    __status_attribute__ = 'run_status'
    __retry_count_attribute__ = 'retry_count'

    def __init__(self, persistence_manager: PersistenceManager, input_path: str):
        self._persistence_manager = persistence_manager
        self._input_path = input_path

    @property
    def processing_start_ts(self) -> int:
        return self.read_attribute(self.__processing_start_ts_attribute__)

    @processing_start_ts.setter
    def processing_start_ts(self, ts: int) -> None:
        self.write_attribute(self.__processing_start_ts_attribute__, ts)

    @property
    def progress(self) -> ProcessorRunProgress:
        raw_value = self.read_attribute(self.__progress_attribute__)
        if raw_value:
            return ProcessorRunProgress(raw_value)
        return ProcessorRunProgress.ENQUEUED

    @progress.setter
    def progress(self, progress: ProcessorRunProgress) -> None:
        self.write_attribute(self.__progress_attribute__, progress.value)

    @property
    def status(self) -> ProcessorRunStatus:
        raw_value = self.read_attribute(self.__status_attribute__)
        if raw_value:
            return ProcessorRunStatus(raw_value)
        return ProcessorRunStatus.UNKNOWN

    @status.setter
    def status(self, status: ProcessorRunStatus) -> None:
        self.write_attribute(self.__status_attribute__, status.value)

    @property
    def retry_count(self) -> int:
        return self.read_attribute(self.__retry_count_attribute__)

    @retry_count.setter
    def retry_count(self, count: int) -> None:
        self.write_attribute(self.__retry_count_attribute__, count)

    def read_attribute(self, attribute: str) -> Any:
        attribute_path = self._persistence_manager.join(self._input_path, '@_' + attribute)
        if self._persistence_manager.exists(attribute_path):
            return self._persistence_manager.get(attribute_path)

    def write_attribute(self, attribute: str, value: Any):
        attribute_path = self._persistence_manager.join(self._input_path, '@_' + attribute)
        self._persistence_manager.set(attribute_path, value)
