# -*- coding: utf-8 -*-
from __future__ import annotations

from collections import OrderedDict
from typing import Dict, List, NamedTuple

from travel.hotels.content_manager.data_model.storage import EntityClass
from travel.hotels.content_manager.data_model.storage import (
    StorageHotelWL, StorageMapping, StoragePermalink, StoragePermalinkWL, StoragePermaroom, StorageSCDescription
)
from travel.hotels.content_manager.data_model.storage import StorageUrl
from travel.hotels.content_manager.data_model.types import uint
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict
from travel.hotels.content_manager.lib.persistence_manager import TableData


class MappingKey(NamedTuple):

    permalink: uint
    operator_id: str
    orig_hotel_id: str
    mapping_key: str

    @staticmethod
    def get_key(mapping: StorageMapping) -> MappingKey:
        return MappingKey(
            mapping.permalink,
            mapping.operator_id,
            mapping.orig_hotel_id,
            mapping.mapping_key,
        )


class UrlKey(NamedTuple):

    permalink: uint
    provider: str

    @staticmethod
    def get_key(url: StorageUrl) -> UrlKey:
        return UrlKey(url.permalink, url.provider)


class HotelWLKey(NamedTuple):

    permalink: uint
    partner_id: str
    original_id: str

    @staticmethod
    def get_key(hotel_wl: StorageHotelWL) -> HotelWLKey:
        return HotelWLKey(
            hotel_wl.permalink,
            hotel_wl.partner_id,
            hotel_wl.original_id,
        )


class SCDescriptionKey(NamedTuple):

    carrier_code: str
    car_type_code: str
    sc_code: str

    @staticmethod
    def get_key(sc_description: StorageSCDescription) -> SCDescriptionKey:
        return SCDescriptionKey(
            sc_description.carrier_code,
            sc_description.car_type_code,
            sc_description.sc_code,
        )


class Storage(object):

    def __init__(self) -> None:
        self.permalinks: Dict[uint, StoragePermalink] = dict()
        self.permarooms: Dict[int, StoragePermaroom] = dict()
        self.mappings: Dict[MappingKey, StorageMapping] = OrderedDict()
        self.urls: Dict[UrlKey, List[StorageUrl]] = dict()

        self.permalinks_wl: Dict[int, StoragePermalinkWL] = dict()
        self.hotels_wl: Dict[HotelWLKey, StorageHotelWL] = dict()

        self.sc_descriptions: Dict[SCDescriptionKey, StorageSCDescription] = dict()

        self._permalink_to_permaroom: Dict[uint, List[int]] = dict()
        self._permalink_to_mapping: Dict[uint, List[MappingKey]] = dict()
        self._permaroom_to_mapping: Dict[int, List[MappingKey]] = dict()
        self._other_mappings: Dict[uint, List[MappingKey]] = dict()
        self._permalink_to_url: Dict[uint, List[UrlKey]] = dict()

        self._permalink_wl_to_hotel: Dict[uint, List[HotelWLKey]] = dict()

    def add_permalink(self, permalink: StoragePermalink) -> None:
        self.permalinks[permalink.id] = permalink

    def add_permaroom(self, permalink: StoragePermalink, permaroom: StoragePermaroom) -> None:
        permaroom_id = permaroom.id
        if permaroom_id is None:
            raise RuntimeError(f'Failed to add permaroom with no id: {permaroom}')
        permaroom.permalink = permalink.id
        self.permarooms[permaroom_id] = permaroom
        permalink_permarooms = self._permalink_to_permaroom.setdefault(permalink.id, list())
        permalink_permarooms.append(permaroom_id)

    def add_mapping(self, permalink: StoragePermalink, mapping: StorageMapping) -> None:
        mapping_key = MappingKey.get_key(mapping)
        self.mappings[mapping_key] = mapping
        permalink_id = permalink.id
        permaroom_id = mapping.permaroom_id
        mapping.permalink = permalink_id
        permalink_mappings = self._permalink_to_mapping.setdefault(permalink_id, list())
        permalink_mappings.append(mapping_key)
        if permaroom_id is not None:
            permaroom_mappings = self._permaroom_to_mapping.setdefault(permaroom_id, list())
            permaroom_mappings.append(mapping_key)
            return
        permalink_other_mappings = self._other_mappings.setdefault(permalink_id, list())
        permalink_other_mappings.append(mapping_key)

    def add_url(self, permalink: StoragePermalink, url: StorageUrl) -> None:
        url_key = UrlKey.get_key(url)
        urls = self.urls.setdefault(url_key, list())
        urls.append(url)
        if url.provider == 'main':
            permalink.hotel_url = url.url
            return
        url.permalink = permalink.id
        permalink_urls = self._permalink_to_url.setdefault(permalink.id, list())
        permalink_urls.append(url_key)

    def add_permalink_wl(self, permalink_wl: StoragePermalinkWL) -> None:
        self.permalinks_wl[permalink_wl.permalink] = permalink_wl

    def add_hotel_wl(self, hotel_wl: StorageHotelWL) -> None:
        hotel_key = HotelWLKey.get_key(hotel_wl)
        permalink_hotels = self._permalink_wl_to_hotel.setdefault(hotel_wl.permalink, list())
        permalink_hotels.append(hotel_key)
        self.hotels_wl[hotel_key] = hotel_wl

    def add_sc_description(self, sc_description: StorageSCDescription) -> None:
        self.sc_descriptions[SCDescriptionKey.get_key(sc_description)] = sc_description

    def apply_data(self, table_data: Dict[EntityClass, TableData], ignore_unknown: bool = False) -> None:
        data = table_data.get(StorageHotelWL)
        if data is not None:
            for row in data:
                hotel_wl = dc_from_dict(StorageHotelWL, row, ignore_unknown)
                self.add_hotel_wl(hotel_wl)

        data = table_data.get(StoragePermalink)
        if data is not None:
            for row in data:
                permalink = dc_from_dict(StoragePermalink, row, ignore_unknown)
                self.add_permalink(permalink)

        data = table_data.get(StoragePermalinkWL)
        if data is not None:
            for row in data:
                permalink_wl = dc_from_dict(StoragePermalinkWL, row, ignore_unknown)
                self.add_permalink_wl(permalink_wl)

        data = table_data.get(StoragePermaroom)
        if data is not None:
            for row in data:
                permaroom = dc_from_dict(StoragePermaroom, row, ignore_unknown)
                self.add_permaroom(self.permalinks[permaroom.permalink], permaroom)

        data = table_data.get(StorageMapping)
        if data is not None:
            for row in data:
                mapping = dc_from_dict(StorageMapping, row, ignore_unknown)
                self.add_mapping(self.permalinks[mapping.permalink], mapping)

        data = table_data.get(StorageUrl)
        if data is not None:
            for row in data:
                url = dc_from_dict(StorageUrl, row, ignore_unknown)
                self.add_url(self.permalinks[url.permalink], url)

        data = table_data.get(StorageSCDescription)
        if data is not None:
            for row in data:
                sc_description = dc_from_dict(StorageSCDescription, row, ignore_unknown)
                self.add_sc_description(sc_description)

    def delete_permaroom(self, permalink: StoragePermalink, permaroom_id: int) -> None:
        permaroom = self.permarooms.get(permaroom_id)
        if permaroom is None:
            return
        permaroom.is_deleted = True

        for mapping_key in self._permaroom_to_mapping.pop(permaroom.id, list()):
            mapping = self.mappings[mapping_key]
            mapping.permaroom_id = None
            self.add_mapping(permalink, mapping)

        try:
            self._permalink_to_permaroom[permalink.id].remove(permaroom.id)
        except (KeyError, ValueError):
            pass

    def delete_other_mapping(self, mapping_key: MappingKey) -> None:
        permalink_id = mapping_key.permalink
        permalink_other_mappings = self._other_mappings.get(permalink_id)
        permalink_other_mappings.remove(mapping_key)
        permalink_mappings = self._permalink_to_mapping.get(permalink_id)
        permalink_mappings.remove(mapping_key)
        self.mappings.pop(mapping_key, None)

    def get_entity_data(self, entity_cls: EntityClass) -> TableData:
        if entity_cls is StorageHotelWL:
            table_data = [dc_to_dict(p) for p in self.hotels_wl.values()]
        elif entity_cls is StoragePermalink:
            table_data = [dc_to_dict(p) for p in self.permalinks.values()]
        elif entity_cls is StoragePermalinkWL:
            table_data = [dc_to_dict(p) for p in self.permalinks_wl.values()]
        elif entity_cls is StoragePermaroom:
            table_data = [dc_to_dict(pr) for pr in self.permarooms.values()]
        elif entity_cls is StorageMapping:
            table_data = [dc_to_dict(m) for m in self.mappings.values()]
        elif entity_cls is StorageUrl:
            table_data = [dc_to_dict(u) for us in self.urls.values() for u in us]
        elif entity_cls is StorageSCDescription:
            table_data = [dc_to_dict(d) for d in self.sc_descriptions.values()]
        else:
            raise Exception(f'Entity not supported {entity_cls}')

        return table_data

    def get_permalink_permarooms(self, permalink: StoragePermalink) -> Dict[int, StoragePermaroom]:
        permarooms = dict()
        for permaroom_id in self._permalink_to_permaroom.get(permalink.id, list()):
            permarooms[permaroom_id] = self.permarooms[permaroom_id]
        return permarooms

    def get_permaroom_mappings(self, permaroom: StoragePermaroom) -> Dict[MappingKey, StorageMapping]:
        mappings = dict()
        for mapping_key in self._permaroom_to_mapping.get(permaroom.id, list()):
            mappings[mapping_key] = self.mappings[mapping_key]
        return mappings

    def get_permalink_mappings(self, permalink: StoragePermalink) -> Dict[MappingKey, StorageMapping]:
        mappings = dict()
        for mapping_key in self._permalink_to_mapping.get(permalink.id, list()):
            mappings[mapping_key] = self.mappings[mapping_key]
        return mappings

    def get_permalink_other_mappings(self, permalink: StoragePermalink) -> Dict[MappingKey, StorageMapping]:
        mappings = dict()
        for mapping_key in self._other_mappings.get(permalink.id, list()):
            mappings[mapping_key] = self.mappings[mapping_key]
        return mappings

    def get_permalink_urls(self, permalink: StoragePermalink) -> Dict[UrlKey, List[StorageUrl]]:
        urls = dict()
        for url_key in self._permalink_to_url.get(permalink.id, list()):
            urls[url_key] = self.urls[url_key]
        return urls

    def permalink_has_permarooms(self, permalink: StoragePermalink) -> bool:
        return permalink.id in self._permalink_to_permaroom

    def get_permalink_wl_hotels(self, permalink: StoragePermalinkWL) -> List[StorageHotelWL]:
        hotels = list()
        for hotel_key in self._permalink_wl_to_hotel.get(permalink.permalink, list()):
            hotels.append(self.hotels_wl[hotel_key])
        return hotels
