# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Callable
import logging

from travel.hotels.content_manager.data_model.storage import TABLE_NAMES
from travel.hotels.content_manager.lib.path_info import PathInfo
from travel.hotels.content_manager.lib.persistence_manager import PersistenceManager


LOG = logging.getLogger(__name__)


class StoragePatcher(object):

    def __init__(self, persistence_manager: PersistenceManager, path_info: PathInfo) -> None:
        self.persistence_manager = persistence_manager
        self.path_info = path_info

    def apply_delete(self, tables_to_delete_path: str, table_name: str) -> None:
        LOG.info('Applying delete patch')
        path_to_remove = self.persistence_manager.join(tables_to_delete_path, table_name)
        LOG.info(f'path_to_remove: {path_to_remove}')
        path_remove_from = self.persistence_manager.join(self.path_info.storage_path, table_name)
        LOG.info(f'path_remove_from: {path_remove_from}')

        dc = TABLE_NAMES.get(table_name)
        if dc is None:
            raise RuntimeError(f'No matching storage table to delete from for {path_to_remove}')

        entities_to_remove = list(self.persistence_manager.read(path_to_remove))
        self.persistence_manager.request_remove(entities_to_remove, path_remove_from, dc)
        LOG.info('Patch applied')

    def apply_upsert(self, tables_to_upsert_path: str, table_name: str) -> None:
        LOG.info('Applying upsert patch')
        src_path = self.persistence_manager.join(tables_to_upsert_path, table_name)
        LOG.info(f'src_path: {src_path}')
        dst_path = self.persistence_manager.join(self.path_info.storage_path, table_name)
        LOG.info(f'dst_path: {dst_path}')

        dc = TABLE_NAMES.get(table_name)
        if dc is None:
            raise RuntimeError(f'No matching storage table to upsert for {src_path}')

        entities_to_upsert = list(self.persistence_manager.read(src_path))
        self.persistence_manager.request_upsert(entities_to_upsert, dst_path, dc)
        LOG.info('Patch applied')

    def apply_patch_by_type(self, patch_path: str, patch_type: str, patch_method: Callable[[str, str], None]) -> bool:
        LOG.info(f'Checking tables for {patch_type}')
        tables_to_patch_path = self.persistence_manager.join(patch_path, patch_type)

        if not self.persistence_manager.exists(tables_to_patch_path):
            LOG.info(f'No such path: {tables_to_patch_path}')
            return False

        tables_to_patch = self.persistence_manager.list(tables_to_patch_path)
        for node in tables_to_patch:
            patch_method(tables_to_patch_path, node.name)

        return len(tables_to_patch) > 0

    def apply_patch(self, patch_path: str) -> bool:
        was_deleted = self.apply_patch_by_type(patch_path, 'delete', self.apply_delete)
        was_upserted = self.apply_patch_by_type(patch_path, 'upsert', self.apply_upsert)
        return was_deleted or was_upserted

    def patch(self) -> bool:
        LOG.info('Patching storage')
        storage_changed = False

        if not self.persistence_manager.exists(self.path_info.storage_patches_path):
            LOG.warning(f'Storage patches path not exists: {self.path_info.storage_patches_path}')
            return storage_changed

        patch_list = self.persistence_manager.list(self.path_info.storage_patches_path)
        for patch_id in patch_list:
            patch_path = self.persistence_manager.join(self.path_info.storage_patches_path, patch_id.name)

            LOG.info(f'Processing patch {patch_path}')
            storage_changed = self.apply_patch(patch_path) or storage_changed

            LOG.info('Deleting patch')
            self.persistence_manager.delete(patch_path)

        if not storage_changed:
            LOG.info('Nothing changed')

        LOG.info('Patching complete')
        return storage_changed
