# -*- coding: utf-8 -*-
from __future__ import annotations

from copy import deepcopy
from enum import Enum
from typing import Any, Dict, NamedTuple
import json
import logging
import requests


LOG = logging.getLogger(__name__)


class ProjectInfo(NamedTuple):
    project_id: str
    css: str
    html: str
    script: str
    other_fields: Dict[str, Any]


class TolokaEnv(Enum):
    TE_TOLOKA_PROD = 'toloka_prod'
    TE_TOLOKA_SANDBOX = 'toloka_sandbox'
    TE_YANG_PROD = 'yang_prod'
    TE_YANG_SANDBOX = 'yang_sandbox'


ENVS = {
    TolokaEnv.TE_TOLOKA_PROD: 'https://toloka.yandex.ru/api/v1',
    TolokaEnv.TE_TOLOKA_SANDBOX: 'https://sandbox.toloka.yandex.ru/api/v1',
    TolokaEnv.TE_YANG_PROD: 'https://yang.yandex-team.ru/api/v1',
    TolokaEnv.TE_YANG_SANDBOX: 'https://sandbox.yang.yandex-team.ru/api/v1',
}


class TolokaClient(object):

    def __init__(self, env: TolokaEnv, token: str):
        self.api_url = ENVS.get(env)
        self.token = token

    @staticmethod
    def has_changes(local: Dict[str, Any], remote: Dict[str, Any]) -> bool:
        for key, local_value in local.items():
            remote_value = remote.get(key)
            if remote_value is None:
                return True
            if local_value != remote_value:
                return True
        return False

    def request(self, method: str, path: str, **kwargs) -> requests.Response:
        headers = {
            'Authorization': f'OAuth {self.token}',
            'Content-Type': 'application/json',
        }
        rsp = requests.request(method, self.api_url + path, headers=headers, **kwargs)
        LOG.info(f'{method} request {rsp.url}')
        rsp.raise_for_status()
        return rsp

    def get_raw_project(self, project_id: str) -> Dict[str, Any]:
        rsp = self.request('GET', f'/projects/{project_id}')
        return json.loads(rsp.text)

    def get_project(self, project_id: str) -> ProjectInfo:
        raw_project = self.get_raw_project(project_id)

        view_spec = raw_project['task_spec']['view_spec']

        return ProjectInfo(
            project_id=project_id,
            css=view_spec.pop('styles'),
            html=view_spec.pop('markup'),
            script=view_spec.pop('script'),
            other_fields=raw_project,
        )

    def update_project(self, project_id: str, css: str, html: str, script: str, other_fields: Dict[str, Any]) -> None:
        remote_params = self.get_raw_project(project_id)
        local_params = deepcopy(remote_params)

        local_params.pop('created', None)
        local_params.update(other_fields)

        view_spec = local_params['task_spec']['view_spec']

        view_spec['styles'] = css
        view_spec['markup'] = html
        view_spec['script'] = script

        if self.has_changes(local_params, remote_params):
            LOG.info(f'Updating project {project_id}')
            self.request('PUT', f'/projects/{project_id}', json=local_params)
            return
        LOG.info('No changes')
