# -*- coding: utf-8 -*-

from argparse import ArgumentParser, Namespace
from uuid import uuid4

from yt.wrapper.client import YtClient

from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.content_manager.lib.persistence_manager import YtPersistenceManager


def request_cl_new_permalinks(persistence_manager: YtPersistenceManager, requests_path: str, options: Namespace):
    request_dir = persistence_manager.join(requests_path, 'wl_start', str(uuid4()))
    request_table = persistence_manager.join(request_dir, 'permalinks')
    persistence_manager.create_dir(request_dir)
    persistence_manager.copy(options.src_table, request_table)

    print(f'Successfully requested for wl_start at {request_table}')


def request_wl_new_hotels(persistence_manager: YtPersistenceManager, requests_path: str, options: Namespace):
    request_dir = persistence_manager.join(requests_path, 'wl_new_hotels', str(uuid4()))
    request_table = persistence_manager.join(request_dir, 'permalinks')
    persistence_manager.create_dir(request_dir)
    persistence_manager.copy(options.src_table, request_table)

    print(f'Successfully requested for wl_new_hotels at {request_table}')


def request_sc_new_descriptions(persistence_manager: YtPersistenceManager, requests_path: str, options: Namespace):
    request_dir = persistence_manager.join(requests_path, 'sc_new_descriptions', str(uuid4()))
    request_table = persistence_manager.join(request_dir, 'descriptions')
    persistence_manager.create_dir(request_dir)
    persistence_manager.copy(options.src_table, request_table)

    print(f'Successfully requested for sc_new_descriptions at {request_table}')


def request_top_permalinks(persistence_manager: YtPersistenceManager, requests_path: str, options: Namespace):
    data = [{'permalink_count': options.count}]
    schema = {'permalink_count': 'uint64'}
    table_path = persistence_manager.join(requests_path, 'top_permalinks', str(uuid4()), 'permalink_count')
    persistence_manager.write(table_path, data, schema)

    print(f'Successfully requested for {options.count} new permalinks at {table_path}')


def request_cr_known_permalinks(persistence_manager: YtPersistenceManager, requests_path: str):
    request_path = persistence_manager.join(requests_path, 'cr_known_permalinks', str(uuid4()))
    persistence_manager.create_dir(request_path)

    print(f'Successfully requested for cr_known_permalinks at {request_path}')


def main():
    parser = ArgumentParser()

    parser.add_argument('--vault-token', default=None)

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--requests-path', required=True)

    subparsers = parser.add_subparsers()

    # cr_known_permalinks
    subparser = subparsers.add_parser('cr_known_permalinks')
    subparser.set_defaults(func=request_cr_known_permalinks)

    # top_permalinks
    subparser = subparsers.add_parser('top_permalinks')
    subparser.set_defaults(func=request_top_permalinks)

    subparser.add_argument('--count', default=100, type=int)

    # wl_start
    subparser = subparsers.add_parser('wl_start')
    subparser.set_defaults(func=request_cl_new_permalinks)

    subparser.add_argument('--src-table', required=True)

    # wl_new_hotels
    subparser = subparsers.add_parser('wl_new_hotels')
    subparser.set_defaults(func=request_wl_new_hotels)

    subparser.add_argument('--src-table', required=True)

    # sc_new_descriptions
    subparser = subparsers.add_parser('sc_new_descriptions')
    subparser.set_defaults(func=request_sc_new_descriptions)

    subparser.add_argument('--src-table', required=True)

    options = parser.parse_args()
    resolve_secrets_ns(options)

    yt_client = YtClient(proxy=options.yt_proxy, token=options.yt_token)
    persistence_manager = YtPersistenceManager(yt_client)

    with persistence_manager.transaction():
        options.func(persistence_manager, options.requests_path, options)


if __name__ == '__main__':
    main()
