# -*- coding: utf-8 -*-
from __future__ import annotations

from enum import Enum
from inspect import isclass
from typing import Any, Callable, Dict, List, Optional
import dataclasses
import logging.config

from travel.library.python.solomon_push_client import SolomonPushClient
from travel.hotels.content_manager.config.stage_config import STAGE_LIST
from travel.hotels.content_manager.data_model import storage
from travel.hotels.content_manager.data_model.options import MetricsOptions
from travel.hotels.content_manager.lib.path_info import PathInfo
from travel.hotels.content_manager.lib.persistence_manager import YtPersistenceManager


LOG = logging.getLogger(__name__)


TableRow = Dict[str, Any]


class Counter(object):

    def __init__(
            self,
            sensor: str,
            labels: Dict[str, str],
            checker: Callable[[TableRow], bool],
    ) -> None:
        self.sensor = sensor
        self.labels = labels
        self.checker = checker
        self.value: int = 0

    def __repr__(self):
        return f'Counter(sensor={self.sensor}, value={self.value}, labels={self.labels})'

    def check(self, row: TableRow):
        if self.checker(row):
            self.value += 1


class ValueCounter(Counter):

    def __init__(
            self,
            sensor: str,
            labels: Dict[str, str],
            field_name: str,
            field_value: Any,
    ):
        self.field_name = field_name
        self.field_value = field_value
        super().__init__(sensor, labels, lambda row: row[self.field_name] == self.field_value)


PERMALINK_COUNTERS = [
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'permalinks',
            'status': 'total',
        },
        checker=lambda row: True,
    ),
]

PERMAROOM_COUNTERS = [
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'permarooms',
            'status': 'total',
        },
        checker=lambda row: True,
    ),
]

MAPPING_COUNTERS = [
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'status': 'total',
        },
        checker=lambda row: True,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'status': 'mapped',
        },
        checker=lambda row: row['permaroom_id'] is not None,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'status': 'not_mapped',
        },
        checker=lambda row: row['permaroom_id'] is None,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'ban_status': 'banned',
        },
        checker=lambda row: row['is_banned'] is True,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'ban_status': 'not_banned',
        },
        checker=lambda row: row['is_banned'] is False,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'need_new_permaroom_check_count': '0',
        },
        checker=lambda row: row['counters'].get('need_new_permaroom', 0) == 0,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'need_new_permaroom_check_count': '1',
        },
        checker=lambda row: row['counters'].get('need_new_permaroom', 0) == 1,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'need_new_permaroom_check_count': '2',
        },
        checker=lambda row: row['counters'].get('need_new_permaroom', 0) == 2,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'need_new_permaroom_check_count': '3',
        },
        checker=lambda row: row['counters'].get('need_new_permaroom', 0) == 3,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'mappings',
            'need_new_permaroom_check_count': '4',
        },
        checker=lambda row: row['counters'].get('need_new_permaroom', 0) == 4,
    ),
]

URLS_COUNTERS = [
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'urls',
            'status': 'total',
        },
        checker=lambda row: True,
    ),
]


HOTELS_WL_COUNTERS = [
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'hotels_wl',
            'status': 'total',
        },
        checker=lambda row: True,
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'hotels_wl',
            'wl_status': 'allowed',
        },
        checker=lambda row: row['is_wl_approved'],
    ),
    Counter(
        sensor='entity_count',
        labels={
            'entity': 'hotels_wl',
            'wl_status': 'not_allowed',
        },
        checker=lambda row: not row['is_wl_approved'],
    ),
]


class MetricsUpdater(object):

    def __init__(
            self,
            persistence_manager: YtPersistenceManager,
            path_info: PathInfo,
            options: MetricsOptions,
            send_metrics_to_solomon: bool,
    ) -> None:
        self.persistence_manager = persistence_manager
        self.path_info = path_info
        self.options = options
        self.send_metrics_to_solomon = send_metrics_to_solomon

        self.solomon_client = SolomonPushClient(
            project=options.project,
            cluster=options.cluster,
            service=options.service,
            token=options.token,
        )

    def get_stage_metrics(self) -> Dict[str, Any]:
        metrics = dict()
        for stage_cfg in STAGE_LIST:
            path = self.persistence_manager.join(self.path_info.stages_path, stage_cfg.name, 'input')
            if not self.persistence_manager.exists(path):
                continue
            job_count = len(self.persistence_manager.list(path))
            metrics[stage_cfg.name] = job_count
        return metrics

    def send_stage_metrics(self, metrics: Dict[str, Any]) -> None:
        for key, value in metrics.items():
            self.solomon_client.send('job_count', value, stage=key)

    @staticmethod
    def get_counters(
        entity_cls: storage.EntityClassWithStatus,
        entity_name: str,
        additional_fields: Optional[List[str]] = None,
    ) -> List[Counter]:
        if additional_fields is None:
            additional_fields = list()
        additional_fields = set(additional_fields)

        counters = list()
        for field in dataclasses.fields(entity_cls):
            field_name = field.name
            field_type = field.type
            if isclass(field_type) and issubclass(field_type, storage.StageStatus):
                stage_name = field_name[len('status_'):]
                for status in storage.StageStatus:
                    counters.append(ValueCounter(
                        sensor='entity_count',
                        labels={
                            'entity': entity_name,
                            'status': status.value,
                            'stage': stage_name,
                        },
                        field_name=field_name,
                        field_value=status.value,
                    ))
            if isclass(field_type) and issubclass(field_type, storage.StageResult):
                stage_name = field_name[:-len('_result')]
                for result in storage.StageResult:
                    counters.append(ValueCounter(
                        sensor='entity_count',
                        labels={
                            'entity': entity_name,
                            'result': result.value,
                            'stage': stage_name,
                        },
                        field_name=field_name,
                        field_value=result.value,
                    ))
            elif field_name in additional_fields:
                if not isclass(field_type) or not issubclass(field_type, Enum):
                    raise Exception(f'Expected Enum for {field_name}')
                for item in field_type:
                    counters.append(ValueCounter(
                        sensor='entity_count',
                        labels={
                            'entity': entity_name,
                            field_name: item.value,
                        },
                        field_name=field_name,
                        field_value=item.value,
                    ))
        return counters

    def calc_metrics(self, path: str, counters: List[Counter]) -> None:
        for row in self.persistence_manager.read(path):
            for counter in counters:
                counter.check(row)

    def send_metrics(self, counters: List[Counter]) -> None:
        for counter in counters:
            self.solomon_client.send(counter.sensor, counter.value, **counter.labels)
            LOG.debug(counter)

    def run(self) -> None:
        LOG.info('Processing permalinks')
        counters = self.get_counters(storage.StoragePermalink, 'permalinks')
        counters.extend(PERMALINK_COUNTERS)
        self.calc_metrics(self.path_info.storage_permalinks_table, counters)
        self.send_metrics(counters)

        LOG.info('Processing permarooms')
        self.calc_metrics(self.path_info.storage_permarooms_table, PERMAROOM_COUNTERS)
        self.send_metrics(PERMAROOM_COUNTERS)

        LOG.info('Processing mappings')
        counters = self.get_counters(storage.StorageMapping, 'mappings')
        counters.extend(MAPPING_COUNTERS)
        self.calc_metrics(self.path_info.storage_mappings_table, counters)
        self.send_metrics(counters)

        LOG.info('Processing urls')
        self.calc_metrics(self.path_info.storage_urls_table, URLS_COUNTERS)
        self.send_metrics(URLS_COUNTERS)

        LOG.info('Processing hotels_wl')
        counters = self.get_counters(storage.StorageHotelWL, 'hotels_wl')
        counters.extend(HOTELS_WL_COUNTERS)
        self.calc_metrics(self.path_info.storage_hotels_wl_table, counters)
        self.send_metrics(counters)

        LOG.info('Processing permalinks_wl')
        counters = self.get_counters(storage.StoragePermalinkWL, 'permalinks')
        self.calc_metrics(self.path_info.storage_permalinks_wl_table, counters)
        self.send_metrics(counters)

        LOG.info('Processing sc_descriptions')
        counters = self.get_counters(
            storage.StorageSCDescription, 'service_class', additional_fields=['sc_description_result']
        )
        self.calc_metrics(self.path_info.storage_sc_descriptions_table, counters)
        self.send_metrics(counters)

        stage_metrics = self.get_stage_metrics()
        LOG.info(f'stage_metrics: {stage_metrics}')
        self.send_stage_metrics(stage_metrics)

        if self.send_metrics_to_solomon:
            self.solomon_client.upload()
            LOG.info('Metrics sent to solomon.')
