# -*- coding: utf-8 -*-

import logging

from travel.hotels.content_manager.data_model.types import AssigneeSkill, ClusterizationProgress, WLResult
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict
from travel.hotels.content_manager.lib.path_info import PathInfo
from travel.hotels.content_manager.lib.persistence_manager import Condition, PersistenceManager
from travel.hotels.content_manager.migrations import storage_21


LOG = logging.getLogger(__name__)


def clusterization_to_actualization(persistence_manager: PersistenceManager, path_info: PathInfo) -> None:
    src_path = path_info.storage_permalinks_wl_table
    LOG.info(f'Getting permalinks from {src_path}')
    raw_permalinks = persistence_manager.request_select(
        src_path=src_path,
        dc=storage_21.StoragePermalinkWL,
        match_conditions=[
            Condition('clusterization_progress', '!=', ClusterizationProgress.NOT_STARTED.value),
        ],
    )
    LOG.info(f'Got {len(raw_permalinks)} to update')

    permalinks = list()
    for row in raw_permalinks:
        permalink: storage_21.StoragePermalinkWL = dc_from_dict(storage_21.StoragePermalinkWL, row)
        permalink.required_stages = 'clusterization'
        permalink.assignee_skill = AssigneeSkill.ADVANCED
        permalink.clusterization_iteration = 1
        permalink.wl_result = WLResult(permalink.clusterization_result.value)
        if permalink.clusterization_progress == ClusterizationProgress.FINISHED:
            permalink.finished_stages = 'clusterization'
        elif permalink.clusterization_progress == ClusterizationProgress.ON_RECHECK:
            permalink.clusterization_iteration = 2
        elif permalink.clusterization_progress == ClusterizationProgress.ON_EDIT:
            permalink.assignee_skill = AssigneeSkill.EDIT
        LOG.info(permalink)
        permalinks.append(dc_to_dict(permalink))

    LOG.info('Updating storage')
    persistence_manager.request_upsert(
        src_data=permalinks,
        dst_path=src_path,
        dc=storage_21.StoragePermalinkWL,
        fields_to_update=[
            'clusterization_progress',
            'required_stages',
            'finished_stages',
            'assignee_skill',
            'clusterization_iteration',
            'wl_result',
        ],
    )


ADDITIONAL_MIGRATIONS = {
    21: clusterization_to_actualization,
}
