# -*- coding: utf-8 -*-

# !!! DO NOT USE "from __future__ import annotations" here
# This will break deserialization from dict

from dataclasses import dataclass, field
from enum import Enum
from typing import Dict, Type, Union

from travel.hotels.content_manager.data_model.common import HotelInfo
from travel.hotels.content_manager.data_model.types import (
    ClusterizationProgress, StageResult, ClusterizationStartReason, uint
)


PERMAROOM_ID_OFFSET = 1_000_000


class StageStatus(Enum):
    NOTHING_TO_DO = 'nothing_to_do'
    TO_BE_PROCESSED = 'to_be_processed'
    IN_PROCESS = 'in_process'


@dataclass
class StoragePermalink:
    id: uint = field(default=0, metadata={'is_key': True})
    is_deleted: bool = False
    hotel_url: str = ''
    marked_by: str = ''
    show_other: bool = True
    original_permalink: uint = 0
    status_partner_offers: StageStatus = StageStatus.NOTHING_TO_DO
    status_partner_offers_ts: uint = 0
    status_yang_rooms: StageStatus = StageStatus.NOTHING_TO_DO
    status_yang_rooms_ts: uint = 0
    status_update_mappings: StageStatus = StageStatus.NOTHING_TO_DO
    status_update_mappings_ts: uint = 0


@dataclass
class StorageHotelWL(HotelInfo):
    permalink: uint = field(default=0, metadata={'is_key': True})
    partner_id: str = field(default='', metadata={'is_key': True})
    original_id: str = field(default='', metadata={'is_key': True})
    is_wl_approved: bool = False
    status_wl_get_hotel_info: StageStatus = StageStatus.NOTHING_TO_DO
    status_wl_get_hotel_info_ts: uint = 0
    status_wl_match_hotels: StageStatus = StageStatus.NOTHING_TO_DO
    status_wl_match_hotels_ts: uint = 0


@dataclass
class StoragePermalinkWL(HotelInfo):
    permalink: uint = field(default=0, metadata={'is_key': True})
    hotel_name: str = ''
    clusterization_start_reason: ClusterizationStartReason = ClusterizationStartReason.COMMON
    clusterization_progress: ClusterizationProgress = ClusterizationProgress.NOT_STARTED
    clusterization_result: StageResult = StageResult.UNKNOWN
    comments: str = ''
    status_clusterization: StageStatus = StageStatus.NOTHING_TO_DO
    status_clusterization_ts: uint = 0


@dataclass
class StoragePermaroom:
    id: int = field(default=None, metadata={'is_key': True})
    name: str = ''
    is_deleted: bool = False
    permalink: uint = field(default=0, metadata={'is_key': True, 'foreign_key': StoragePermalink.id})
    comment: str = ''
    alternative_names: str = ''


@dataclass
class StorageMapping:
    is_deleted: bool = False
    is_hidden: bool = False
    is_banned: bool = False
    need_new_permaroom: bool = False
    permalink: uint = field(default=0, metadata={'is_key': True, 'foreign_key': StoragePermalink.id})
    operator_id: str = field(default='', metadata={'is_key': True})
    orig_hotel_id: str = field(default='', metadata={'is_key': True})
    mapping_key: str = field(default='', metadata={'is_key': True})
    permaroom_id: int = field(default=None, metadata={'foreign_key': StoragePermaroom.id})
    orig_room_name: str = ''
    count: int = 0
    prices_per_night: str = ''
    requests_range: str = ''
    price: int = 0
    url: str = ''
    params: str = ''
    comment: str = ''
    status_yang_mappings: StageStatus = StageStatus.NOTHING_TO_DO
    status_yang_mappings_ts: uint = 0
    status_offer_prioritization: StageStatus = StageStatus.NOTHING_TO_DO
    status_offer_prioritization_ts: uint = 0
    counters: Dict[str, int] = field(default_factory=dict)


@dataclass
class StorageUrl:
    permalink: uint = field(default=0, metadata={'is_key': True, 'foreign_key': StoragePermalink.id})
    provider: str = field(default='', metadata={'is_key': True})
    url: str = field(default='', metadata={'is_key': True})


TABLE_NAMES = {
    'permalinks': StoragePermalink,
    'permarooms': StoragePermaroom,
    'mappings': StorageMapping,
    'urls': StorageUrl,
    'hotels_wl': StorageHotelWL,
    'permalinks_wl': StoragePermalinkWL,
}


CatroomEntityClass = Union[Type[StoragePermalink], Type[StoragePermaroom], Type[StorageMapping], Type[StorageUrl]]
WhitelistEntityClass = Union[Type[StorageHotelWL], Type[StoragePermalinkWL]]
EntityClass = Union[CatroomEntityClass, WhitelistEntityClass]


CatroomEntityClassWithStatus = Union[Type[StoragePermalink], Type[StorageMapping]]
WhitelistEntityClassWithStatus = Union[Type[StorageHotelWL], Type[StoragePermalinkWL]]
EntityClassWithStatus = Union[CatroomEntityClassWithStatus, WhitelistEntityClassWithStatus]


CatroomEntityWithStatus = Union[StoragePermalink, StorageMapping]
WhitelistEntityWithStatus = Union[StorageHotelWL, StoragePermalinkWL]
EntityWithStatus = Union[CatroomEntityWithStatus, WhitelistEntityWithStatus]
