# -*- coding: utf-8 -*-

import logging.config
from argparse import ArgumentParser, Namespace
from datetime import datetime

from yt.wrapper.client import YtClient

from travel.library.python.time_interval import TimeInterval
from travel.library.python.tools import replace_args_from_env
from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.content_manager.config.stage_config import (
    ActualizationStageConfig,
    CallCenterStageConfig,
    CatroomExportStageConfig,
    ClusterizationStageConfig,
    CrKnownPermalinksStageConfig,
    OfferPrioritizationStageConfig,
    PartnerOffersStageConfig,
    SCExportDescriptionsStageConfig,
    SCNewDescriptionsStageConfig,
    SCUpdateDescriptionsStageConfig,
    TopPermalinksStageConfig,
    UpdateMappingsStageConfig,
    WLClusterizedHotelsStageConfig,
    WLExportStageConfig,
    WLGetHotelInfoStageConfig,
    WLMatchHotelsStageConfig,
    WLNewHotelsStageConfig,
    WLStartStageConfig,
    YangMappingsStageConfig,
    YangRoomsStageConfig,
)
from travel.hotels.content_manager.lib.common import hide_secrets_ns
from travel.hotels.content_manager.lib.path_info import PathInfo
from travel.hotels.content_manager.lib.processor import Processor
from travel.hotels.content_manager.lib.processor_run_info import (
    ProcessorRunInfo, ProcessorRunProgress, ProcessorRunStatus
)
from travel.hotels.content_manager.lib.persistence_manager import YtPersistenceManager
from travel.hotels.content_manager.lib.yql_simple_client import YqlSimpleClient
from travel.hotels.content_manager.processors_exec.log_config import LOG_CONFIG


LOG = logging.getLogger(__name__)


class ProcessorRunner:

    def __init__(self, options: Namespace):
        self.processing_start_ts = int(datetime.utcnow().timestamp())

        self.yql_client = YqlSimpleClient(token=options.yql_token, yt_proxy=options.yt_proxy)
        self.yt_client = YtClient(proxy=options.yt_proxy, token=options.yt_token)
        self.persistence_manager = YtPersistenceManager(self.yt_client, self.yql_client)

        self.options = options

    def run(self) -> None:
        input_path = self.options.input_path
        stage_input_path, job_id = self.persistence_manager.split(input_path)
        stage_path = self.persistence_manager.dirname(stage_input_path)
        output_path = self.persistence_manager.join(stage_path, 'output', job_id)

        processor_storage_path = self.persistence_manager.join(input_path, 'storage')
        path_info = PathInfo(
            persistence_manager=self.persistence_manager,
            root=self.options.yt_root,
            storage_path=processor_storage_path,
        )

        stage = self.options.stage
        processor_cls = stage.mock if self.options.use_mock else stage.processor

        if processor_cls is None:
            raise Exception(f'No suitable processor/mock for {stage}')

        processor = processor_cls(
            persistence_manager=self.persistence_manager,
            yql_client=self.yql_client,
            path_info=path_info,
            input_path=input_path,
            output_path=output_path,
            job_id=job_id,
            options=self.options,
        )

        processor_run_info = ProcessorRunInfo(self.persistence_manager, processor.input_path)
        processor_run_info.processing_start_ts = self.processing_start_ts
        processor_run_info.progress = ProcessorRunProgress.RUNNING
        try:
            self.try_run(processor)
            processor_run_info.status = ProcessorRunStatus.SUCCESS
        except Exception as e:
            logging.error('Failed to run processor', exc_info=e)
            processor_run_info.status = ProcessorRunStatus.FAILED
        processor_run_info.progress = ProcessorRunProgress.FINISHED

    def try_run(self, processor: Processor) -> None:
        with self.persistence_manager.transaction() as t:
            if not self.persistence_manager.lock(processor.input_path):
                raise RuntimeError(f'Failed to get exclusive lock on {processor.input_path}')
            self.yql_client.transaction_id = t.transaction_id
            processor.process()
            self.yql_client.transaction_id = None
            processor.post_process()


def main():
    logging.config.dictConfig(LOG_CONFIG)

    parser = ArgumentParser()

    parser.add_argument('--vault-token', default=None)

    parser.add_argument('--use-mock', action='store_true')

    parser.add_argument('--yql-token', required=True)

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)

    parser.add_argument(
        '--yt-root',
        required=True,
        help='//home/travel/${USER}/content_manager/catroom for dev/testing'
    )

    parser.add_argument(
        '--temp-dir',
        required=True,
        help='//home/travel/${USER}/content_manager/catroom/temp for dev/testing'
    )

    parser.add_argument('--input-path', required=True)

    subparsers = parser.add_subparsers(dest='task_name')

    subparser = subparsers.add_parser(TopPermalinksStageConfig.name)
    subparser.set_defaults(stage=TopPermalinksStageConfig)
    subparser.add_argument('--days', default=10)

    subparser = subparsers.add_parser(OfferPrioritizationStageConfig.name)
    subparser.set_defaults(stage=OfferPrioritizationStageConfig)
    subparser.add_argument('--ignore-skipped', action='store_true')
    subparser.add_argument('--no-filter', action='store_true')
    subparser.add_argument('--other-percentage', default=0.1, type=float)
    subparser.add_argument('--probability', default=0.95, type=float)

    subparser = subparsers.add_parser(CrKnownPermalinksStageConfig.name)
    subparser.set_defaults(stage=CrKnownPermalinksStageConfig)

    subparser = subparsers.add_parser(UpdateMappingsStageConfig.name)
    subparser.set_defaults(stage=UpdateMappingsStageConfig)

    subparser = subparsers.add_parser(PartnerOffersStageConfig.name)
    subparser.set_defaults(stage=PartnerOffersStageConfig)

    subparser = subparsers.add_parser(YangRoomsStageConfig.name)
    subparser.set_defaults(stage=YangRoomsStageConfig)

    subparser = subparsers.add_parser(YangMappingsStageConfig.name)
    subparser.set_defaults(stage=YangMappingsStageConfig)

    subparser = subparsers.add_parser(CatroomExportStageConfig.name)
    subparser.set_defaults(stage=CatroomExportStageConfig)
    subparser.add_argument('--config-dir', required=True)

    subparser = subparsers.add_parser(WLStartStageConfig.name)
    subparser.set_defaults(stage=WLStartStageConfig)

    subparser = subparsers.add_parser(ActualizationStageConfig.name)
    subparser.set_defaults(stage=ActualizationStageConfig)

    subparser = subparsers.add_parser(CallCenterStageConfig.name)
    subparser.set_defaults(stage=CallCenterStageConfig)
    subparser.add_argument('--altay-url', default='')
    subparser.add_argument('--altay-user-id', default='')
    subparser.add_argument('--priority', default=95)
    subparser.add_argument('--call-center-queue', default='ytravel')
    subparser.add_argument('--check-intrerval', type=TimeInterval, default=TimeInterval('15m'))

    subparser = subparsers.add_parser(ClusterizationStageConfig.name)
    subparser.set_defaults(stage=ClusterizationStageConfig)

    subparser = subparsers.add_parser(WLClusterizedHotelsStageConfig.name)
    subparser.set_defaults(stage=WLClusterizedHotelsStageConfig)

    subparser = subparsers.add_parser(WLExportStageConfig.name)
    subparser.set_defaults(stage=WLExportStageConfig)
    subparser.add_argument('--direct-boy-export-path', required=True)
    subparser.add_argument('--export-path', required=True)
    subparser.add_argument('--transfer-to', nargs='*', default=['arnold'])

    subparser = subparsers.add_parser(WLGetHotelInfoStageConfig.name)
    subparser.set_defaults(stage=WLGetHotelInfoStageConfig)

    subparser = subparsers.add_parser(WLMatchHotelsStageConfig.name)
    subparser.set_defaults(stage=WLMatchHotelsStageConfig)

    subparser = subparsers.add_parser(WLNewHotelsStageConfig.name)
    subparser.set_defaults(stage=WLNewHotelsStageConfig)

    subparser = subparsers.add_parser(SCExportDescriptionsStageConfig.name)
    subparser.set_defaults(stage=SCExportDescriptionsStageConfig)
    subparser.add_argument('--export-path', required=True)

    subparser = subparsers.add_parser(SCNewDescriptionsStageConfig.name)
    subparser.set_defaults(stage=SCNewDescriptionsStageConfig)

    subparser = subparsers.add_parser(SCUpdateDescriptionsStageConfig.name)
    subparser.set_defaults(stage=SCUpdateDescriptionsStageConfig)

    options = parser.parse_args(replace_args_from_env())
    LOG.debug(f'Unresolved options: {options}')
    resolve_secrets_ns(options)
    LOG.debug(f'Resolved options: {hide_secrets_ns(options)}')

    ProcessorRunner(options).run()


if __name__ == '__main__':
    main()
