# -*- coding: utf-8 -*-
from __future__ import annotations

import logging
import random
from datetime import datetime
from enum import Enum
from uuid import uuid4

from travel.hotels.content_manager.data_model.stage import ActualizationData, ActualizationDataOutputResult
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)

PERMALINK_ID_MAX = 2 ** 64 - 1


class FailedChoices(Enum):
    FAILED_CLOSED = 'failed_closed'
    FAILED_TEMPORARY_CLOSED = 'failed_temporary_closed'
    FAILED_UNPUBLISHED = 'failed_unpublished'
    FAILED_DENIED_PUBLISHING = 'failed_denied_publishing'
    FAILED_TECHNICAL_FAULT = 'failed_technical_fault'
    FAILED_NO_HOTEL_RUBRIC = 'failed_no_hotel_rubric'
    FAILED_CHECK = 'failed_check'
    FAILED_OTHER = 'failed_other'


class MockActualization(Processor):

    @staticmethod
    def rand_bool(probability: float):
        return random.random() < probability

    def set_actualization_result(self, task: ActualizationData) -> None:

        task_input_permalink = int(task.input.permalink)
        task.output.comments = f'comments_{task_input_permalink}'

        if self.rand_bool(.8):
            result = ActualizationDataOutputResult.SUCCESS
            task.output.checked_attributes = task.input.required_attributes
        elif self.rand_bool(.8):
            result = ActualizationDataOutputResult.FAILED
            choice = random.choice([item for item in FailedChoices])
            setattr(task.output, choice.value, True)
        elif self.rand_bool(.8):
            result = ActualizationDataOutputResult.STAGE_CALL_CENTER_REQUIRED
        elif self.rand_bool(.8):
            result = ActualizationDataOutputResult.STAGE_CLUSTERIZATION_REQUIRED
        else:
            result = ActualizationDataOutputResult.STAGE_ACTUALIZATION_REQUIRED
            attributes = list()
            for attribute in task.input.required_attributes:
                if self.rand_bool(.95):
                    attributes.append(attribute)
            task.output.checked_attributes = attributes
        task.output.result = result

    @staticmethod
    def set_info(task: ActualizationData, start_ts: int) -> None:
        pool = random.choice([1, 2])
        info = task.info
        info.assignment_id = str(uuid4())
        info.create_ts = start_ts
        info.pool_id = str(pool)
        info.reward = float(pool)
        info.status = 'ACCEPTED'
        info.submit_ts = start_ts + random.randint(300, 900)
        info.worker_id = str(random.randint(0, 10))

    def run(self):
        LOG.info(f'Processing {self.input_path}')
        random.seed()

        start_ts = int(datetime.utcnow().timestamp())

        input_table = self.persistence_manager.join(self.input_path, 'assignments')
        output_table = self.persistence_manager.join(self.output_path, 'assignments')

        output_data = list()

        for row in self.persistence_manager.read(input_table):
            task: ActualizationData = dc_from_dict(ActualizationData, row)
            self.set_actualization_result(task)
            self.set_info(task, start_ts)
            output_data.append(dc_to_dict(task))

        LOG.info(f'Processed {len(output_data)} tasks')

        LOG.info(f'Write result to {output_table}')
        self.persistence_manager.write(output_table, output_data, get_dc_yt_schema(ActualizationData))

        LOG.info('All done')
