# -*- coding: utf-8 -*-
from __future__ import annotations

from itertools import product
from typing import Iterable
import logging

from travel.hotels.content_manager.data_model.stage import ActualizationData
from travel.hotels.content_manager.data_model.stage import ActualizationDataInput, ActualizationDataOutput
from travel.hotels.content_manager.data_model.stage import PermalinkTaskInfo
from travel.hotels.content_manager.data_model.storage import StoragePermalinkWL
from travel.hotels.content_manager.data_model.options import TriggerOptions
from travel.hotels.content_manager.data_model.types import AssigneeSkill, ActualizationStartReason
from travel.hotels.content_manager.lib.attributes import Attributes
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.trigger import EntityGroupingKey, Producer, ThreadFilter, Trigger


LOG = logging.getLogger(__name__)


class ProducerActualization(Producer):

    def prepare_data(
            self,
            trigger: Trigger,
            thread_filter: ThreadFilter,
            local_storage: Storage,
            path: str,
            entities: Iterable[StoragePermalinkWL],
            grouping_key: EntityGroupingKey,
            options: TriggerOptions,
    ):
        LOG.info('Preparing trigger data')
        task_data = list()

        for permalink in entities:
            LOG.debug(permalink)

            if permalink.comments is None:
                permalink.comments = ''
            required_attributes = sorted(a.value for a in Attributes.get_attributes_to_check(permalink))

            cd_input = ActualizationDataInput(
                permalink=str(permalink.permalink),
                altay_url=f'https://altay.yandex-team.ru/cards/perm/{permalink.permalink}',
                required_attributes=required_attributes,
                requirements=permalink.requirements.split('\n'),
                prev_comments=permalink.comments.split('\n'),
                hotel_name=permalink.hotel_name,
                assignee_skill=permalink.assignee_skill,
            )
            cd = ActualizationData(
                input=cd_input,
                output=ActualizationDataOutput(),
                info=PermalinkTaskInfo(),
            )
            task_data.append(dc_to_dict(cd))

        hotels_table = trigger.persistence_manager.join(path, 'assignments')
        LOG.info(f'Writing result to {hotels_table}')
        trigger.persistence_manager.write(hotels_table, task_data, get_dc_yt_schema(ActualizationData))

        trigger.write_options(options, hotels_table, grouping_key)


def get_filter(start_reason: ActualizationStartReason, assignee_skill: AssigneeSkill) -> ThreadFilter:
    options_id = f'{start_reason.value}_{assignee_skill.value}'

    # noinspection PyUnusedLocal
    def func_first(entity: StoragePermalinkWL, storage: Storage) -> bool:
        return (
            entity.actualization_start_reason == start_reason and
            entity.assignee_skill == assignee_skill
        )

    return ThreadFilter(options_id, options_id, func_first)


FILTERS_ACTUALIZATION = [get_filter(p, r) for p, r in product(ActualizationStartReason, AssigneeSkill)]
