# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import (
    ActualizationData, ActualizationDataOutputResult, PermalinkTaskInfo
)
from travel.hotels.content_manager.data_model.types import ActualizableAttribute, StageResult
from travel.hotels.content_manager.lib.common import dc_from_dict, str_from_set, str_to_set
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class UpdaterActualization(Updater):

    __result_map__ = {
        ActualizationDataOutputResult.SUCCESS: StageResult.SUCCESS,
        ActualizationDataOutputResult.STAGE_CALL_CENTER_REQUIRED: StageResult.SUCCESS,
        ActualizationDataOutputResult.FAILED: StageResult.FAILED,
    }

    def get_result_data(self, output_path: str) -> List[ActualizationData]:
        output_data = list()
        output_table = self.persistence_manager.join(output_path, 'assignments')
        LOG.info(f'Reading processor results from {output_table}')
        for row in self.persistence_manager.read(output_table):
            actualization_data: ActualizationData = dc_from_dict(ActualizationData, row, ignore_unknown=True)
            output_data.append(actualization_data)
        return output_data

    def get_local_storage(self, output_data: List[ActualizationData]) -> Storage:
        permalink_ids = set()
        for item in output_data:
            permalink_ids.add(int(item.input.permalink))

        raw_permalinks = self.persistence_manager.request_select(
            self.path_info.storage_permalinks_wl_table,
            StoragePermalinkWL,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )

        table_data = {StoragePermalinkWL: raw_permalinks}
        local_storage = Storage()
        local_storage.apply_data(table_data)
        return local_storage

    def apply_results(self, results: Iterable[ActualizationData], local_storage: Storage) -> None:
        permalinks_finished = list()
        permalinks_in_process = list()

        for item in results:
            permalink_id = item.input.permalink
            permalink = local_storage.permalinks_wl[int(permalink_id)]

            permalink.assignee_skill = item.output.assignee_skill
            permalink.comments = item.output.comments

            checked_attributes = str_to_set(permalink.checked_attributes, lambda x: ActualizableAttribute(x))
            checked_attributes |= set(item.output.checked_attributes)
            permalink.checked_attributes = str_from_set(checked_attributes, lambda x: x.value)

            if item.output.result in (
                ActualizationDataOutputResult.SUCCESS,
                ActualizationDataOutputResult.FAILED,
            ):
                permalink.actualization_result = self.__result_map__[item.output.result]
                permalink.actualization_iteration = 0
                permalink.date_next_check = item.output.date_next_check
                permalinks_finished.append(permalink)
            elif item.output.result == ActualizationDataOutputResult.STAGE_ACTUALIZATION_REQUIRED:
                permalinks_in_process.append(permalink)
                permalink.actualization_iteration += 1
            elif item.output.result == ActualizationDataOutputResult.STAGE_CLUSTERIZATION_REQUIRED:
                permalink.actualization_iteration += 1
                self.dispatcher.add_stage_required_stages(permalink, self.stage_name, ['clusterization'])
            elif item.output.result == ActualizationDataOutputResult.STAGE_CALL_CENTER_REQUIRED:
                self.dispatcher.add_stage_required_stages(permalink, self.stage_name, ['call_center'])
                permalink.actualization_iteration += 1
            else:
                raise Exception(f'Unexpected result {item.output.result}')

        self.mark_as_processed(StoragePermalinkWL, local_storage.permalinks_wl.values())
        self.send_to_stage(StoragePermalinkWL, permalinks_in_process, self.stage_name)
        self.update_finished_stages(permalinks_finished)
        self.dispatch_entities(StoragePermalinkWL, local_storage.permalinks_wl.values(), 'wl_clusterized_hotels')

    def get_task_logs(self, hotels: Iterable[ActualizationData]) -> List[PermalinkTaskInfo]:
        logs = list()
        for result in hotels:
            task_info = result.info
            task_info.stage = self.stage_name
            task_info.permalink = int(result.input.permalink)
            task_info.reward = float(task_info.reward)
            logs.append(task_info)
        return logs

    def run(self, output_path: str, temp_dir: str) -> None:
        result_data = self.get_result_data(output_path)
        LOG.info(f'Got {len(result_data)} result records')

        local_storage = self.get_local_storage(result_data)
        self.apply_results(result_data, local_storage)

        self.add_fields_to_update(
            StoragePermalinkWL,
            [
                'actualization_result',
                'actualization_iteration',
                'checked_attributes',
                'assignee_skill',
                'required_stages',
                'finished_stages',
                'actualization_required_stages',
                'date_next_check',
                'comments',
            ],
        )
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalinkWL), StoragePermalinkWL)

        self.update_task_logs(self.get_task_logs(result_data), PermalinkTaskInfo, 'permalinks')
