import logging

from dataclasses import dataclass

from typing import List, Optional

from enum import Enum

from requests import Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util import Retry

from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict


class TaskStatus(Enum):
    IN_WORK = 'in_work'
    COMPLETED_WITH_ANSWERED_CALL = 'completed_with_answered_call'
    COMPLETED_WITHOUT_ANSWERED_CALL = 'completed_without_answered_call'


#  Документация https://github.yandex-team.ru/mining/altay2/blob/master/callcenter-main/swagger/callcenter.json
@dataclass
class PermalinkWithPriority:
    companies: List[int]
    priority: int


@dataclass
class ActualizationHotelsRequest:
    companies: List[int]
    type: str
    permalinks_with_priority: List[PermalinkWithPriority]
    comment: str


@dataclass
class CompanyIdWithRequestId:
    company_id: Optional[int] = None
    permalink: Optional[int] = None
    request_id: Optional[int] = None


@dataclass
class ActualizationHotelsResult:
    created: Optional[List[CompanyIdWithRequestId]] = None
    failed: Optional[List[int]] = None


@dataclass
class TaskStatusResult:
    status: TaskStatus


class AltayClient:

    def __init__(self, base_url: str, user_id: int):
        if base_url.endswith('/'):
            base_url = base_url[:-1]
        self.base_url = base_url
        self.user_id = user_id

    def actualize_hotels(self, request: ActualizationHotelsRequest) -> ActualizationHotelsResult:
        url = self.base_url + '/request/submit/actualize-companies'
        session = self._get_session()
        data = dc_to_dict(request)
        response = session.post(url, params=dict(user=self.user_id), json=data)
        response.raise_for_status()
        result_dict = response.json()
        logging.info(f'Got response: {result_dict}')
        return dc_from_dict(ActualizationHotelsResult, result_dict)

    def get_task_status(self, request_id: int) -> TaskStatusResult:
        url = f'{self.base_url}/request/{request_id}/task-status'
        session = self._get_session()
        response = session.get(url, params=dict(user=self.user_id))
        response.raise_for_status()
        return dc_from_dict(TaskStatusResult, response.json())

    def _get_session(self):
        session = Session()
        retry = Retry(
            total=15,
            method_whitelist=frozenset(['POST']),
            backoff_factor=5.0,
            status_forcelist=frozenset([500, 503, 413, 429])
        )
        session.mount('http://', HTTPAdapter(max_retries=retry))
        session.mount('https://', HTTPAdapter(max_retries=retry))
        return session
