from dataclasses import dataclass
from typing import List, Iterable
import logging

from travel.hotels.content_manager.data_model.stage import (
    CallCenterData, CallCenterDataInput, CallCenterDataOutput, CallCenterDataOutputResult
)
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.processor import Processor

from travel.hotels.content_manager.stages.call_center.altay_client import (
    ActualizationHotelsRequest, TaskStatus, AltayClient, PermalinkWithPriority,
)

LOG = logging.getLogger(__name__)


@dataclass
class ProcessingInfo:
    input: CallCenterDataInput
    output: CallCenterDataOutput


class ProcessorCallCenter(Processor):
    altay_client: AltayClient

    def run(self) -> None:
        self.altay_client = AltayClient(self.options.altay_url, self.options.altay_user_id)
        assignments = self._get_assignments()
        assignments_to_request = filter(lambda el: el.input.request_id is None,
                                        assignments)
        assignments_to_check = filter(lambda el: el.input.request_id is not None,
                                      assignments)
        self._request_processing(assignments_to_request)
        self._check_request_status(assignments_to_check)
        self._save_results(assignments)

    def _get_assignments(self) -> List[ProcessingInfo]:
        input_path = self.persistence_manager.join(
            self.input_path,
            'assignments',
        )
        LOG.info(f'Getting assignments from {input_path}')
        assignments = list()
        for raw_assignment in self.persistence_manager.read(input_path):
            assignment = dc_from_dict(ProcessingInfo, raw_assignment)
            assignments.append(assignment)
        return assignments

    def _check_request_status(self, assignments_to_check: Iterable[ProcessingInfo]):
        for assignment in assignments_to_check:
            try:
                response = self.altay_client.get_task_status(assignment.input.request_id)
                if response.status is None:
                    LOG.exception(f'None status for {response}')
                    assignment.output.result = CallCenterDataOutputResult.FAILED
                if response.status == TaskStatus.IN_WORK:
                    assignment.output.result = CallCenterDataOutputResult.IN_PROCESS
                if response.status == TaskStatus.COMPLETED_WITH_ANSWERED_CALL:
                    assignment.output.result = CallCenterDataOutputResult.SUCCESS_WITH_CALL
                if response.status == TaskStatus.COMPLETED_WITHOUT_ANSWERED_CALL:
                    assignment.output.result = CallCenterDataOutputResult.SUCCESS_WITHOUT_CALL
            except Exception as e:
                LOG.exception(f'Check status failed: {e}')
                assignment.output.result = CallCenterDataOutputResult.FAILED

    def _request_processing(self, assignments_to_request: Iterable[ProcessingInfo]):
        for assignment in assignments_to_request:
            request = self._get_actualization_request(assignment)
            try:
                response = self.altay_client.actualize_hotels(request)
                if len(response.failed) > 0:
                    raise Exception('failed to create request at call center')

                assignment.output.request_id = response.created[0].request_id
                assignment.output.result = CallCenterDataOutputResult.IN_PROCESS
            except Exception as e:
                LOG.exception(f'Failed to request actualization for {assignment.input.permalink}', exc_info=e)
                assignment.output.result = CallCenterDataOutputResult.FAILED

    def _get_actualization_request(self, assignment_to_request: ProcessingInfo) -> ActualizationHotelsRequest:
        comment_list = [assignment_to_request.input.comments]
        comment_list.extend(assignment_to_request.input.required_attributes)
        comment = '\n'.join(comment_list)
        permalinks = [PermalinkWithPriority(
            companies=[assignment_to_request.input.permalink],
            priority=self.options.priority,
        )]
        request: ActualizationHotelsRequest = ActualizationHotelsRequest(
            companies=[],
            comment=comment,
            type=self.options.call_center_queue,
            permalinks_with_priority=permalinks,
        )
        return request

    def _save_results(self, assignments: List[ProcessingInfo]) -> None:
        path = self.persistence_manager.join(self.output_path, 'assignments')
        raw_assignments = list()
        for assignment in assignments:
            ccd = CallCenterData(
                input=assignment.input,
                output=assignment.output,
            )
            raw_assignments.append(dc_to_dict(ccd))

        with self.persistence_manager.transaction():
            self.persistence_manager.write(path, raw_assignments, get_dc_yt_schema(CallCenterData))
