# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import CallCenterData, CallCenterDataOutputResult
from travel.hotels.content_manager.data_model.types import StageResult
from travel.hotels.content_manager.lib.common import dc_from_dict
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.updater import Updater

LOG = logging.getLogger(__name__)


class UpdaterCallCenter(Updater):
    __result_map__ = {
        CallCenterDataOutputResult.SUCCESS_WITH_CALL: StageResult.SUCCESS,
        CallCenterDataOutputResult.FAILED: StageResult.FAILED,
        CallCenterDataOutputResult.IN_PROCESS: StageResult.UNKNOWN,
        CallCenterDataOutputResult.SUCCESS_WITHOUT_CALL: StageResult.SUCCESS,
        CallCenterDataOutputResult.UNKNOWN: StageResult.FAILED,
    }

    def get_result_data(self, output_path: str) -> List[CallCenterData]:
        output_data = list()
        output_table = self.persistence_manager.join(output_path, 'assignments')
        LOG.info(f'Reading processor results from {output_table}')
        for row in self.persistence_manager.read(output_table):
            call_center_data: CallCenterData = dc_from_dict(CallCenterData, row, ignore_unknown=True)
            output_data.append(call_center_data)
        return output_data

    def get_local_storage(self, output_data: List[CallCenterData]) -> Storage:
        permalink_ids = set()
        for item in output_data:
            permalink_ids.add(int(item.input.permalink))

        raw_permalinks = self.persistence_manager.request_select(
            self.path_info.storage_permalinks_wl_table,
            StoragePermalinkWL,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )

        table_data = {StoragePermalinkWL: raw_permalinks}
        local_storage = Storage()
        local_storage.apply_data(table_data)
        return local_storage

    def apply_results(self, results: Iterable[CallCenterData], local_storage: Storage) -> None:
        permalinks_in_work: List[StoragePermalinkWL] = list()
        permalinks_finished: List[StoragePermalinkWL] = list()
        for item in results:
            permalink_id = item.input.permalink
            permalink = local_storage.permalinks_wl[int(permalink_id)]
            permalink.call_center_result = self.__result_map__[item.output.result]
            if item.output.request_id is not None:
                permalink.call_center_request_id = item.output.request_id
            if item.output.result == CallCenterDataOutputResult.IN_PROCESS:
                permalink.call_center_iteration += 1
                permalinks_in_work.append(permalink)
            if item.output.result == CallCenterDataOutputResult.FAILED:
                permalink.call_center_iteration += 1
                permalinks_in_work.append(permalink)
            if item.output.result == CallCenterDataOutputResult.SUCCESS_WITH_CALL:
                permalink.call_center_iteration = 0
                permalink.call_center_request_id = None
                message = 'оператор КЦ дозвонился до отеля'
                permalink.comments += message if permalink.comments == '' else f'\n{message}'
                permalinks_finished.append(permalink)
            if item.output.result == CallCenterDataOutputResult.SUCCESS_WITHOUT_CALL:
                permalink.call_center_iteration = 0
                permalink.call_center_request_id = None
                message = 'оператор КЦ не дозвонился до отеля'
                permalink.comments += message if permalink.comments == '' else f'\n{message}'
                permalinks_finished.append(permalink)
            if item.output.result == CallCenterDataOutputResult.UNKNOWN:
                raise Exception(f'Unknown result for {permalink=}')
        LOG.info(f'{len(permalinks_in_work)=}')
        LOG.info(f'{len(permalinks_finished)=}', )
        self.mark_as_processed(StoragePermalinkWL, local_storage.permalinks_wl.values())
        self.update_finished_stages(permalinks_finished)
        self.send_to_stage(StoragePermalinkWL, permalinks_in_work, self.stage_name)
        self.dispatch_entities(StoragePermalinkWL, permalinks_finished, 'wl_clusterized_hotels')

    def run(self, output_path: str, temp_dir: str) -> None:
        result_data = self.get_result_data(output_path)
        LOG.info(f'Got {len(result_data)} result records')

        local_storage = self.get_local_storage(result_data)
        self.apply_results(result_data, local_storage)

        self.add_fields_to_update(
            StoragePermalinkWL,
            [
                'call_center_result',
                'checked_attributes',
                'required_stages',
                'finished_stages',
                'call_center_request_id',
                'call_center_iteration',
                'comments',
            ],
        )
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalinkWL), StoragePermalinkWL)
