import logging

from travel.hotels.content_manager.data_model.storage import PERMAROOM_ID_OFFSET
from travel.hotels.content_manager.lib.processor import Processor

LOG = logging.getLogger(__name__)


class ProcessorCatroomExport(Processor):
    DEFAULT_PERMAROOM_ID = 1
    HIDDEN_PERMAROOM_ID = PERMAROOM_ID_OFFSET - 1

    def __init__(self, **kwargs):
        super(ProcessorCatroomExport, self).__init__(**kwargs)
        self.config_dir = self.options.config_dir

    def run(self) -> None:
        operators_table = self.persistence_manager.join(self.config_dir, 'operators')

        export_path = self.persistence_manager.join(self.path_info.root, 'export', 'catroom')

        export_catroom_permalinks_table = self.persistence_manager.join(export_path, 'permalinks')
        export_catroom_permarooms_table = self.persistence_manager.join(export_path, 'permarooms')
        export_catroom_mappings_table = self.persistence_manager.join(export_path, 'mappings')

        self.yql_client.run_query(
            f"""
                /* Копирование permalinks */
                INSERT INTO
                    `{export_catroom_permalinks_table}`
                WITH TRUNCATE
                SELECT
                    id, show_other
                FROM
                    `{self.path_info.storage_permalinks_table}`
                ;

                /* Копирование permarooms */
                INSERT INTO
                    `{export_catroom_permarooms_table}`
                WITH TRUNCATE
                SELECT
                    id, name, false as is_hidden, false as is_other
                FROM
                    `{self.path_info.storage_permarooms_table}`
                WHERE
                    is_deleted = false
                ;
                INSERT INTO
                    `{export_catroom_permarooms_table}` (id, name, is_hidden, is_other)
                VALUES
                    ({self.DEFAULT_PERMAROOM_ID}, "Другие", false, true), /* Дефолтный пермарум */
                    ({self.HIDDEN_PERMAROOM_ID}, "Скрытый", true, false) /* Пермарум для скрытия маппингов */
                ;

                /* Копирование mappings */
                INSERT INTO
                    `{export_catroom_mappings_table}`
                WITH TRUNCATE
                SELECT
                    Ensure(o.OperatorIdInt, o.OperatorIdInt IS NOT NULL) AS operator_id,
                    m.orig_hotel_id AS orig_hotel_id,
                    m.mapping_key AS mapping_key,
                    m.permalink AS permalink,
                    CASE
                        WHEN m.is_hidden = false
                        THEN m.permaroom_id
                        ELSE {self.HIDDEN_PERMAROOM_ID} /* Указываем id специального permaroom */
                    END as permaroom_id
                FROM `{self.path_info.storage_mappings_table}` AS m
                LEFT JOIN `{operators_table}` AS o
                ON m.operator_id == o.OperatorId
                LEFT JOIN `{self.path_info.storage_permarooms_table}` AS permarooms
                ON m.permaroom_id == permarooms.id
                WHERE
                    (m.permaroom_id IS NOT NULL AND NOT permarooms.is_deleted)
                    OR m.is_hidden
                ;
            """,
        )
        LOG.info("CatRoom export processor successfully finished")
