# -*- coding: utf-8 -*-
from __future__ import annotations

from datetime import datetime
from enum import Enum
from typing import List
from uuid import uuid4
import logging
import random

from travel.hotels.content_manager.data_model.stage import ClusterizationData, ClusterizationDataOutputResult
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)

PERMALINK_ID_MAX = 2 ** 64 - 1


class FailedChoices(Enum):
    FAILED_NO_HOTEL_RUBRIC = 'failed_no_hotel_rubric'
    FAILED_UNPUBLISHED = 'failed_unpublished'
    FAILED_DENIED_PUBLISHING = 'failed_denied_publishing'
    FAILED_TECHNICAL_FAULT = 'failed_technical_fault'
    FAILED_CLOSED = 'failed_closed'
    FAILED_TEMPORARY_CLOSED = 'failed_temporary_closed'
    FAILED_OTHER = 'failed_other'


class MockClusterization(Processor):

    @staticmethod
    def rand_bool(probability: float):
        return random.random() < probability

    def get_random_permalink(self, permalink: int, permalink_ids: List[int]) -> str:
        result = permalink
        if self.rand_bool(.1):
            result = random.randint(0, PERMALINK_ID_MAX)
        if self.rand_bool(.1):
            result = random.choice(permalink_ids)
        return str(result)

    def set_clusterization_result(self, task: ClusterizationData, permalink_ids: List[int]) -> None:
        result = random.choice(
            [item for item in ClusterizationDataOutputResult if item != ClusterizationDataOutputResult.UNKNOWN]
        )
        task.output.result = result

        task_input_permalink = int(task.input.permalink)
        task.output.permalink = self.get_random_permalink(task_input_permalink, permalink_ids)
        task.output.stage_actualization_required = self.rand_bool(.1)
        task.output.stage_call_center_required = self.rand_bool(.1)
        task.output.comments = f'comments_{task_input_permalink}'

        if result == ClusterizationDataOutputResult.FAILED:
            choice = random.choice([item for item in FailedChoices])
            setattr(task.output, choice.value, True)

    @staticmethod
    def set_info(task: ClusterizationData, start_ts: int) -> None:
        pool = random.choice([1, 2])
        info = task.info
        info.assignment_id = str(uuid4())
        info.create_ts = start_ts
        info.pool_id = str(pool)
        info.reward = float(pool)
        info.status = 'ACCEPTED'
        info.submit_ts = start_ts + random.randint(300, 900)
        info.worker_id = str(random.randint(0, 10))

    def run(self):
        LOG.info(f'Processing {self.input_path}')
        random.seed()

        start_ts = int(datetime.utcnow().timestamp())

        input_table = self.persistence_manager.join(self.input_path, 'hotels')
        output_table = self.persistence_manager.join(self.output_path, 'hotels')

        raw_input = self.persistence_manager.read(input_table)

        tasks = list()
        permalink_ids = list()
        output_data = list()

        for row in raw_input:
            task: ClusterizationData = dc_from_dict(ClusterizationData, row)
            tasks.append(task)
            permalink_ids.append(int(task.input.permalink))

        for task in tasks:
            self.set_clusterization_result(task, permalink_ids)
            self.set_info(task, start_ts)
            output_data.append(dc_to_dict(task))

        LOG.info(f'Processed {len(output_data)} permalinks')

        LOG.info(f'Write result to {output_table}')
        self.persistence_manager.write(output_table, output_data, get_dc_yt_schema(ClusterizationData))

        LOG.info('All done')
