# -*- coding: utf-8 -*-
from __future__ import annotations

from copy import deepcopy
from typing import Any, Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StoragePermalinkWL, StageStatus
from travel.hotels.content_manager.data_model.stage import (
    ClusterizationData, ClusterizationDataOutputResult, PermalinkTaskInfo
)
from travel.hotels.content_manager.data_model.types import StageResult
from travel.hotels.content_manager.lib.attributes import Attributes
from travel.hotels.content_manager.lib.common import dc_from_dict, str_to_set
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)

OUTDATED_START_REASONS = {
    'no_hotel_rubric',
    'hotel_apartment',
    'unpublished',
    'season_hotels',
    'correction',
    'common_20_6_2',
}


class UpdaterClusterization(Updater):

    def get_result_data(self, output_path: str) -> List[ClusterizationData]:
        output_data = list()
        output_table = self.persistence_manager.join(output_path, 'hotels')
        LOG.info(f'Reading processor results from {output_table}')
        clusterization_data: ClusterizationData
        for row in self.persistence_manager.read(output_table):
            clusterization_data = dc_from_dict(ClusterizationData, row, ignore_unknown=True)
            new_permalinks = clusterization_data.output.new_permalinks
            if new_permalinks is None:
                new_permalinks = list()
            new_permalinks = list(filter(lambda x: bool(x.strip()), new_permalinks))
            clusterization_data.output.new_permalinks = new_permalinks
            try:
                int(clusterization_data.output.permalink)
                for permalink in clusterization_data.output.new_permalinks:
                    int(permalink)
            except Exception as e:
                LOG.error(e)
                clusterization_data.output.permalink = clusterization_data.input.permalink
            output_data.append(clusterization_data)
        return output_data

    def get_local_storage(self, output_data: List[ClusterizationData]) -> Storage:
        permalink_ids = set()
        for item in output_data:
            permalink_ids.add(int(item.input.permalink))
            permalink_ids.add(int(item.output.permalink))
            for permalink in item.output.new_permalinks:
                permalink_ids.add(int(permalink))

        raw_permalinks = self.persistence_manager.request_select(
            self.path_info.storage_permalinks_wl_table,
            StoragePermalinkWL,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )
        for permalink in raw_permalinks:
            if permalink['clusterization_start_reason'] in OUTDATED_START_REASONS:
                permalink['clusterization_start_reason'] = 'common'

        table_data = {StoragePermalinkWL: raw_permalinks}
        local_storage = Storage()
        local_storage.apply_data(table_data)
        return local_storage

    def clone_permalink(
        self,
        original: StoragePermalinkWL,
        new_ids: List[str],
        local_storage: Storage,
        **fields_to_update: Any,
    ) -> List[StoragePermalinkWL]:
        clones = list()
        for permalink_id in new_ids:
            permalink_id = int(permalink_id)
            permalink = local_storage.permalinks_wl.get(permalink_id)
            if permalink is not None:
                required_stages = str_to_set(permalink.required_stages)
                finished_stages = str_to_set(permalink.finished_stages)
                stages_not_finished = required_stages - finished_stages
                if stages_not_finished:
                    permalink.comments = original.comments
                    self.dispatcher.add_entity_required_stages(permalink, [self.stage_name])
                    permalink.priority = max(permalink.priority, original.priority)
                    running_stages = self.dispatcher.get_entity_running_stages(permalink)
                    if running_stages:
                        running_stages -= {self.stage_name}
                        self.dispatcher.add_stage_required_stages(permalink, self.stage_name, running_stages)
                clones.append(permalink)
                continue
            permalink = deepcopy(original)
            permalink.permalink = permalink_id
            for name, value in fields_to_update.items():
                setattr(permalink, name, value)
            local_storage.add_permalink_wl(permalink)
            clones.append(permalink)
        return clones

    def apply_results(self, results: Iterable[ClusterizationData], local_storage: Storage) -> None:
        permalinks_finished = list()
        permalinks_in_process = list()

        for item in results:
            input_permalink_id = item.input.permalink
            output_permalink_id = item.output.permalink
            input_permalink = local_storage.permalinks_wl[int(input_permalink_id)]
            if input_permalink_id == output_permalink_id:
                permalink = input_permalink
            else:
                clones = self.clone_permalink(input_permalink, [output_permalink_id], local_storage)
                if not clones:
                    continue
                permalink = clones[0]
                permalink.hotel_name = ''
                input_permalink.required_stages = ''
                input_permalink.clusterization_result = StageResult.UNKNOWN
                input_permalink.clusterization_iteration = 0
            self.mark_as_processed(StoragePermalinkWL, [input_permalink])

            permalink.required_attributes = Attributes.allowed_attributes
            permalink.assignee_skill = item.output.assignee_skill
            required_stages = list()
            if item.output.stage_actualization_required and item.output.stage_call_center_required:
                required_stages.append('call_center')
                self.dispatcher.add_stage_required_stages(permalink, 'call_center', ['actualization'])
            elif item.output.stage_actualization_required:
                required_stages.append('actualization')
            elif item.output.stage_call_center_required:
                required_stages.append('call_center')
            self.dispatcher.add_entity_required_stages(permalink, required_stages)
            permalink.date_next_check = item.output.date_next_check
            permalink.comments = item.output.comments

            new_permalink_ids = item.output.new_permalinks or list()
            clones = self.clone_permalink(
                original=permalink,
                new_ids=new_permalink_ids,
                local_storage=local_storage,
                clusterization_iteration=1,
                status_clusterization=StageStatus.NOTHING_TO_DO,
                status_clusterization_ts=self.start_ts,
                route='',
            )
            for clone in clones:
                self.dispatcher.add_entity_required_stages(clone, [self.stage_name])
            self.dispatch_entities_delayed(StoragePermalinkWL, clones, 'wl_clusterized_hotels')

            if item.output.result == ClusterizationDataOutputResult.SUCCESS:
                permalink.clusterization_result = StageResult.SUCCESS
                permalink.clusterization_iteration = 0
                permalinks_finished.append(permalink)
            elif item.output.result == ClusterizationDataOutputResult.FAILED:
                permalink.clusterization_result = StageResult.FAILED
                permalink.clusterization_iteration = 0
                permalinks_finished.append(permalink)
            elif item.output.result == ClusterizationDataOutputResult.IN_PROCESS:
                permalink.clusterization_iteration += 1
                permalinks_in_process.append(permalink)

        self.send_to_stage(StoragePermalinkWL, permalinks_in_process, self.stage_name)
        self.mark_as_processed(StoragePermalinkWL, permalinks_finished)
        self.update_finished_stages(permalinks_finished)
        self.dispatch_entities(StoragePermalinkWL, permalinks_finished, 'wl_clusterized_hotels')

    def get_task_logs(self, hotels: Iterable[ClusterizationData]) -> List[PermalinkTaskInfo]:
        logs = list()
        for result in hotels:
            task_info = result.info
            task_info.stage = self.stage_name
            task_info.permalink = int(result.output.permalink)
            task_info.reward = float(task_info.reward)
            logs.append(task_info)
        return logs

    def run(self, output_path: str, temp_dir: str) -> None:
        result_data = self.get_result_data(output_path)
        LOG.info(f'Got {len(result_data)} result records')

        local_storage = self.get_local_storage(result_data)
        self.apply_results(result_data, local_storage)

        self.add_fields_to_update(
            StoragePermalinkWL,
            [
                'clusterization_result',
                'clusterization_iteration',
                'priority',
                'assignee_skill',
                'required_stages',
                'finished_stages',
                'call_center_required_stages',
                'required_attributes',
                'date_next_check',
                'comments',
                'route',
                'hotel_name',
            ],
        )
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalinkWL), StoragePermalinkWL)

        self.update_task_logs(self.get_task_logs(result_data), PermalinkTaskInfo, 'permalinks')
