import logging

from travel.hotels.content_manager.data_model.storage import StageStatus, StoragePermalink, StoragePermaroom
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.processor import Processor
from travel.hotels.content_manager.lib.storage import Storage

LOG = logging.getLogger(__name__)

ENTITIES_TO_READ = [
    StoragePermalink,
    StoragePermaroom,
]


class ProcessorCrKnownPermalinks(Processor):

    def run(self) -> None:
        local_storage_path = self.persistence_manager.join(self.output_path, 'local_storage')
        local_storage = Storage()

        storage_permalinks_table = self.path_info.storage_permalinks_table

        LOG.info(f'Getting permalinks from {storage_permalinks_table}')
        raw_permalinks = self.persistence_manager.request_select(
            src_path=storage_permalinks_table,
            dc=StoragePermalink,
            match_conditions=[
                Condition('status_partner_offers', '==', StageStatus.NOTHING_TO_DO.value) &
                Condition('status_yang_rooms', '==', StageStatus.NOTHING_TO_DO.value) &
                Condition('status_update_mappings', '==', StageStatus.NOTHING_TO_DO.value) &
                Condition('is_deleted', '==', False)
            ],
        )

        permalinks = [StoragePermalink(**p) for p in raw_permalinks]
        LOG.info(f'Got {len(permalinks)} permalinks to process')

        permalink_ids = list({p.id for p in permalinks})

        storage_permarooms_table = self.path_info.storage_permarooms_table

        LOG.info(f'Getting permarooms from {storage_permarooms_table}')
        raw_permarooms = self.persistence_manager.request_select(
            src_path=storage_permarooms_table,
            dc=StoragePermaroom,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )

        local_storage.apply_data({StoragePermalink: raw_permalinks, StoragePermaroom: raw_permarooms})

        result_permalinks = (
            dc_to_dict(p) for p in local_storage.permalinks.values() if local_storage.permalink_has_permarooms(p)
        )
        result_table = self.persistence_manager.join(self.output_path, 'permalinks')

        LOG.info(f'Writing result to {result_table}')
        self.persistence_manager.write(result_table, result_permalinks, get_dc_yt_schema(StoragePermalink))

        LOG.info(f'Removing {local_storage_path}')
        self.persistence_manager.delete(local_storage_path)
