# -*- coding: utf-8 -*-
from __future__ import annotations

from datetime import datetime
from uuid import uuid4
import logging
import random

from travel.hotels.content_manager.data_model.stage import SCUpdateDescriptionsData
from travel.hotels.content_manager.data_model.types import SCDescriptionResult
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)

DATA_SOURCE_MAX = 5
DESCRIPTION_ID_MAX = 2 ** 64 - 1
SPECIFIC_COUNT_MAX = 3
TRAIN_NUMBER_MAX = 500


ALLOWED_RESULTS = [
    SCDescriptionResult.UPDATED,
    SCDescriptionResult.NO_DATA,
    SCDescriptionResult.ACTUAL,
]


class MockSCUpdateDescriptions(Processor):

    @staticmethod
    def set_result(task: SCUpdateDescriptionsData) -> None:
        result = random.choice(ALLOWED_RESULTS)
        task.output.result = result
        if result != SCDescriptionResult.UPDATED:
            return

        task.output.is_source_official = random.choice([True, False])
        task.output.data_source = f'data_source_{random.randint(0, DATA_SOURCE_MAX)}'
        task.output.sc_description = f'sc_description_{random.randint(0, DESCRIPTION_ID_MAX)}'
        task.output.sc_description_original = f'sc_description_original_{random.randint(0, DESCRIPTION_ID_MAX)}'
        specific_count = random.randint(0, SPECIFIC_COUNT_MAX)
        sc_description_specific = list()
        for _ in range(specific_count):
            tn = f'{random.randint(0, TRAIN_NUMBER_MAX)}|{random.randint(0, TRAIN_NUMBER_MAX)}'
            ds = f'description_specific_{random.randint(0, TRAIN_NUMBER_MAX)}'
            sc_description_specific.append({'car_numbers': tn, 'description': ds})
        task.output.sc_description_specific = sc_description_specific

    @staticmethod
    def set_info(task: SCUpdateDescriptionsData, start_ts: int) -> None:
        pool = random.choice([1, 2])
        info = task.info
        info.assignment_id = str(uuid4())
        info.create_ts = start_ts
        info.pool_id = str(pool)
        info.reward = float(pool)
        info.status = 'ACCEPTED'
        info.submit_ts = start_ts + random.randint(300, 900)
        info.worker_id = str(random.randint(0, 10))

    def run(self):
        LOG.info(f'Processing {self.input_path}')
        random.seed()

        start_ts = int(datetime.utcnow().timestamp())

        input_table = self.persistence_manager.join(self.input_path, 'descriptions')
        output_table = self.persistence_manager.join(self.output_path, 'descriptions')

        raw_input = self.persistence_manager.read(input_table)

        output_data = list()
        for row in raw_input:
            task: SCUpdateDescriptionsData = dc_from_dict(SCUpdateDescriptionsData, row)
            self.set_result(task)
            self.set_info(task, start_ts)
            output_data.append(dc_to_dict(task))

        LOG.info(f'Processed {len(output_data)} descriptions')

        LOG.info(f'Write result to {output_table}')
        self.persistence_manager.write(output_table, output_data, get_dc_yt_schema(SCUpdateDescriptionsData))

        LOG.info('All done')
