# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StageStatus, StorageSCDescription
from travel.hotels.content_manager.data_model.stage import DescriptionTaskInfo, SCUpdateDescriptionsData
from travel.hotels.content_manager.data_model.types import SCDescriptionResult
from travel.hotels.content_manager.lib.common import dc_from_dict
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.storage import SCDescriptionKey, Storage
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class UpdaterSCUpdateDescriptions(Updater):

    @staticmethod
    def get_key(description: SCUpdateDescriptionsData) -> SCDescriptionKey:
        return SCDescriptionKey(
            description.input.carrier_code,
            description.input.car_type_code,
            description.input.sc_code,
        )

    def get_results(self, output_path: str) -> List[SCUpdateDescriptionsData]:
        results = list()

        descriptions_table = self.persistence_manager.join(output_path, 'descriptions')
        LOG.info(f'Reading descriptions from {descriptions_table}')

        for row in self.persistence_manager.read(descriptions_table):
            dd = dc_from_dict(SCUpdateDescriptionsData, row)
            results.append(dd)

        return results

    def update_local_storage(self, results: Iterable[SCUpdateDescriptionsData], local_storage: Storage) -> None:
        processed_scs = [dd.input.sc_code for dd in results]
        descriptions = self.persistence_manager.request_select(
            src_path=self.path_info.storage_sc_descriptions_table,
            dc=StorageSCDescription,
            match_conditions=[
                Condition('sc_code', 'in', processed_scs) &
                Condition('status_sc_update_descriptions', '==', StageStatus.IN_PROCESS.value),
            ],
        )
        table_data = {StorageSCDescription: descriptions}
        local_storage.apply_data(table_data)

        processed_descriptions = list()
        for dd in results:
            key = self.get_key(dd)
            existing_description = local_storage.sc_descriptions.get(key, None)

            if existing_description is None:
                raise Exception(f'Failed to get description from local store for {dd}')

            df = existing_description
            df.sc_description_result = dd.output.result
            if df.sc_description_result == SCDescriptionResult.UPDATED:
                df.data_source = dd.input.url if dd.output.is_source_official else dd.output.data_source
                df.sc_description = dd.output.sc_description
                df.sc_description_original = dd.output.sc_description_original
                df.sc_description_specific = dd.output.sc_description_specific

            processed_descriptions.append(existing_description)

        self.mark_as_processed(StorageSCDescription, processed_descriptions)

    def get_task_logs(self, hotels: Iterable[SCUpdateDescriptionsData]) -> List[DescriptionTaskInfo]:
        logs = list()
        for result in hotels:
            task_info = result.info
            task_info.stage = self.stage_name
            task_info.carrier_code = result.input.carrier_code
            task_info.car_type_code = result.input.car_type_code
            task_info.sc_code = result.input.sc_code
            task_info.reward = float(task_info.reward)
            logs.append(task_info)
        return logs

    def run(self, output_path: str, temp_dir: str) -> None:
        local_storage = Storage()

        results = self.get_results(output_path)
        LOG.info(f'Got {len(results)} descriptions')
        self.update_task_logs(self.get_task_logs(results), DescriptionTaskInfo, 'sc_descriptions')

        LOG.info('Adding descriptions to storage')
        self.update_local_storage(results, local_storage)
        fields_to_update = [
            'sc_description_result',
            'data_source',
            'sc_description',
            'sc_description_original',
            'sc_description_specific',
        ]
        self.add_fields_to_update(StorageSCDescription, fields_to_update)
        self.update_storage_entity(local_storage.get_entity_data(StorageSCDescription), StorageSCDescription)
