# -*- coding: utf-8 -*-

from datetime import date

from dateutil.relativedelta import relativedelta

from travel.hotels.content_manager.lib.processor import Processor


class ProcessorTopPermalinks(Processor):

    def get_popular_hotels(self, known_permalinks_table, output_table, permalink_count):
        date_to = date.today()
        date_from = date_to - relativedelta(days=self.options.days)
        date_from = date_from.isoformat()
        date_to = date_to.isoformat()
        query = f'''
            USE {self.options.yt_proxy};
            PRAGMA yt.InferSchema;
            PRAGMA yson.DisableStrict;

            $get_permalinks = ($x) -> {{
                $list_data = Yson::ConvertToList($x);
                $permalinks_only = ListMap($list_data, ($y) -> {{ RETURN Yson::LookupString($y, "permalink"); }});
                $filtered_data = ListFilter($permalinks_only, ($y) -> {{ RETURN $y IS NOT NULL; }});
                RETURN ListUniq(ListMap($filtered_data, ($y) -> {{ RETURN CAST($y AS Uint64); }}));
            }};

            $get_travel_data = ($x) -> {{
                RETURN ListFlatMap(Yson::ConvertToList($x),
                    ($y) -> {{RETURN IF(Yson::ConvertToString($y["path"]) LIKE "%travel_map%" OR
                    Yson::ConvertToString($y["path"]) LIKE "%travel_company%", 1, NULL)}});
            }};

            $clicked_permalinks = (
                SELECT
                    $get_permalinks(clicks) as permalinks
                FROM RANGE(`home/travel/analytics/user_sessions`, "{date_from}", "{date_to}")
                WHERE
                    ListLength($get_travel_data(clicks)) > 0
                    AND ListLength($get_permalinks(clicks)) > 0
                    AND device == "desktop"
            );

            $flat_clicked = (
                SELECT
                    permalink
                FROM $clicked_permalinks
                FLATTEN LIST BY permalinks as permalink
            );

            $popular_hotels = (
                SELECT
                    permalink,
                    COUNT(*) as popularity
                FROM $flat_clicked
                GROUP BY permalink
            );

            $new_popular_hotels = (
                SELECT
                    popular_hotels.permalink AS permalink,
                    popular_hotels.popularity AS popularity
                FROM $popular_hotels AS popular_hotels
                LEFT ONLY JOIN `{known_permalinks_table}` AS known_permalinks
                ON (popular_hotels.permalink == known_permalinks.id)
            );

            $hotels_filtered = (
                SELECT
                    t1.permalink AS permalink,
                    t1.popularity AS popularity
                FROM $new_popular_hotels AS t1
                INNER JOIN `//home/travel/prod/general/altay_mappings/latest/permalink_to_hotel_info` AS t2
                ON (t1.permalink == t2.permalink)
                WHERE
                    t2.is_platinum = TRUE AND
                    t2.country_code == "RU" AND
                    ListIndexOf(rubric_permalinks ?? AsList(""), "184106414") IS NOT NULL
            );

            $top_hotels = (
                SELECT permalink, popularity
                    FROM $hotels_filtered
                    ORDER BY popularity DESC
                    LIMIT {permalink_count}
            );

            INSERT INTO `{output_table}` WITH TRUNCATE
            SELECT permalink
            FROM $top_hotels;
        '''
        self.yql_client.run_query(query)

    def run(self):
        input_table = self.persistence_manager.join(self.input_path, 'permalink_count')
        permalink_count = list(self.persistence_manager.read(input_table))[0]['permalink_count']
        output_table = self.persistence_manager.join(self.output_path, 'permalinks')
        known_permalinks_table = self.path_info.storage_permalinks_table
        self.get_popular_hotels(known_permalinks_table, output_table, permalink_count)
