# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Dict, Iterable
import logging

from travel.hotels.content_manager.data_model.stage import WLNewHotelsPermalinkData
from travel.hotels.content_manager.data_model.storage import StoragePermalinkWL
from travel.hotels.content_manager.data_model.options import TriggerOptions
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.dispatcher import Dispatcher
from travel.hotels.content_manager.lib.persistence_manager import PersistenceManager
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.trigger import EntityGroupingKey, Producer, ThreadFilter, Trigger

LOG = logging.getLogger(__name__)


class ProducerWLClusterizedHotels(Producer):

    @staticmethod
    def write_permalinks(
        persistence_manager: PersistenceManager,
        table_path: str,
        permalinks: Iterable[WLNewHotelsPermalinkData],
        schema: Dict[str, str],
    ):
        LOG.info(f'Writing permalinks to {table_path}')
        raw_permalinks = (dc_to_dict(p) for p in permalinks)
        persistence_manager.write(table_path, raw_permalinks, schema)

    def prepare_data(
            self,
            trigger: Trigger,
            thread_filter: ThreadFilter,
            local_storage: Storage,
            path: str,
            entities: Iterable[StoragePermalinkWL],
            grouping_key: EntityGroupingKey,
            options: TriggerOptions,
    ):
        success_permalinks_table = trigger.persistence_manager.join(path, 'permalinks')
        failed_permalinks_table = trigger.persistence_manager.join(path, 'failed_permalinks')

        success_permalinks = list()
        failed_permalinks = list()

        for storage_permalink in entities:
            stage_permalink = WLNewHotelsPermalinkData(
                permalink=storage_permalink.permalink,
                grouping_key=grouping_key.group_name,
            )
            if Dispatcher.is_processing_successful(storage_permalink):
                success_permalinks.append(stage_permalink)
            else:
                failed_permalinks.append(stage_permalink)

        schema = get_dc_yt_schema(WLNewHotelsPermalinkData)
        self.write_permalinks(trigger.persistence_manager, success_permalinks_table, success_permalinks, schema)
        self.write_permalinks(trigger.persistence_manager, failed_permalinks_table, failed_permalinks, schema)
